/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.util;

import java.util.Map;

/**
 * utility class for counter aggregation.
 */
public final class Counters {
  /** prevent instance creation. */
  public static final double NANOS_PER_SEC = 1e9;

  /** name for counters and durations. */
  // CHECKSTYLE:OFF
  public static final String WARN_COUNT = "warnCount";

  public static final String INPUT = "input";

  public static final String OUTPUT = "output";

  public static final String FUNCTION = "function";

  public static final String DURATION = "duration";

  public static final String DURATION_IODATA = DURATION + ".iodata";

  public static final String DURATION_IODATA_OPEN = DURATION_IODATA + ".open";

  public static final String DURATION_IODATA_CLOSE = DURATION_IODATA + ".close";

  public static final String DURATION_PERFORM = DURATION + ".perform";

  public static final String DURATION_PERFORM_FUNCTION = DURATION_PERFORM + "." + FUNCTION;

  public static final String DURATION_PERFORM_INPUT = DURATION_PERFORM + "." + INPUT;

  public static final String DURATION_PERFORM_OUTPUT = DURATION_PERFORM + "." + OUTPUT;

  public static final String DATAOBJECTCOUNT = "dataObjectCount";
  // CHECKSTYLE:ON

  /** prevent instance creation. */
  private Counters() {
    throw new IllegalStateException("prevent instance creation");
  }

  /** convert nanoseconds value to seconds (as double) and add to counter key. */
  public static void addDuration(final Map<String, Number> counters, final String key, final long nanoSeconds) {
    final double duration = nanoSeconds / NANOS_PER_SEC;
    add(counters, key, duration);
  }

  /** add count value to counter key. An existing counter value is converted to a long value. */
  public static void add(final Map<String, Number> counters, final String key, final long count) {
    if (counters.containsKey(key)) {
      counters.put(key, counters.get(key).longValue() + count);
    } else {
      counters.put(key, count);
    }
  }

  /** add count value to counter key. An existing counter value is converted to a double value. */
  public static void add(final Map<String, Number> counters, final String key, final double count) {
    if (counters.containsKey(key)) {
      counters.put(key, counters.get(key).doubleValue() + count);
    } else {
      counters.put(key, count);
    }
  }

  /** add all counters from the second map to the first one. */
  public static void addAll(final Map<String, Number> counters, final Map<String, Number> moreCounters) {
    for (final Map.Entry<String, Number> entry : moreCounters.entrySet()) {
      final String key = entry.getKey();
      final Number value = entry.getValue();
      if (value instanceof Double) {
        add(counters, key, value.doubleValue());
      } else {
        add(counters, key, value.longValue());
      }
    }
  }

  /** add all counters from the second map to the first one, add a prefix to the names of the added counters. */
  public static void addAll(final Map<String, Number> counters, final Map<String, Number> moreCounters,
    final String prefix) {
    for (final Map.Entry<String, Number> entry : moreCounters.entrySet()) {
      final String key = entry.getKey();
      final Number value = entry.getValue();
      if (value instanceof Double) {
        add(counters, prefix + "." + key, value.doubleValue());
      } else {
        add(counters, prefix + "." + key, value.longValue());
      }
    }
  }
}
