/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Drazen Cindric (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.processing.httphandler;

import java.net.MalformedURLException;
import java.util.List;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.WorkflowProcessor;

/**
 * Implements the handling of HTTP requests for reading pipeline definitions.
 * 
 * @author drazen
 * 
 */
public class PipelinesHandler extends JsonRequestHandler {

  /** Exception message if no definition has been found. */
  public static final String EXCEPTION_MESSAGE = "There is no definition for given pipeline";

  /**
   * associated workflow processor.
   */
  private WorkflowProcessor _processor;

  /**
   * {@inheritDoc}
   * 
   * PipelineListHandler supports GET.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "POST".equals(method);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord,
    final HttpExchange exchange) throws Exception {
    final String requestHost = getRequestHost(exchange);
    if ("POST".equals(method)) {
      return definePipeline(requestHost, requestUri, inputRecord);
    } else {
      return getPipelineList(requestHost, requestUri);
    }
  }

  /** create list of pipelines descriptions. */
  private Object getPipelineList(final String requestHost, final String requestUri) throws MalformedURLException {
    final AnyMap result = FACTORY.createAnyMap();
    final AnySeq pipelines = FACTORY.createAnySeq();
    final List<String> names = _processor.getWorkflowNames();
    for (final String name : names) {
      final AnyMap pipeline = FACTORY.createAnyMap();
      pipeline.put(WorkflowProcessor.WORKFLOW_NAME, name);
      pipeline.put("url", URLCreator.create(requestHost, requestUri, name).toString());
      pipelines.add(pipeline);
    }
    result.put("pipelines", pipelines);
    return result;
  }

  /** define or update a pipeline definition. */
  private Object definePipeline(final String requestHost, final String requestUri, final Record inputRecord)
    throws ProcessingException, MalformedURLException {
    if (inputRecord == null) {
      throw new IllegalArgumentException("Empty request not allowed.");
    }
    final AnyMap definition = inputRecord.getMetadata();
    final String workflowName = definition.getStringValue(WorkflowProcessor.WORKFLOW_NAME);
    _processor.setWorkflowDefinition(workflowName, definition);
    final AnyMap result = DataFactory.DEFAULT.createAnyMap();
    result.put(WorkflowProcessor.WORKFLOW_NAME, workflowName);
    if (definition.containsKey(WorkflowProcessor.WORKFLOW_TIMESTAMP)) {
      result.put(WorkflowProcessor.WORKFLOW_TIMESTAMP, definition.get(WorkflowProcessor.WORKFLOW_TIMESTAMP));
    }
    result.put("url", URLCreator.create(requestHost, requestUri, workflowName).toExternalForm());
    return result;
  }

  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof ProcessingException) {
      return ((ProcessingException) ex).isRecoverable() ? HttpStatus.INTERNAL_SERVER_ERROR : HttpStatus.BAD_REQUEST;
    } else if (ex instanceof IllegalArgumentException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * result status for POST requests should be CREATED.
   * 
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    if ("POST".equals(method)) {
      return HttpStatus.CREATED;
    }
    return super.getSuccessStatus(method, requestUri);
  }

  /**
   * set workflow processor reference.
   * 
   * @param processor
   *          workflow processor
   */
  public void setProcessor(final WorkflowProcessor processor) {
    _processor = processor;
  }

  /**
   * remove workflow processor reference.
   * 
   * @param processor
   *          workflow processor
   */
  public void unsetProcessor(final WorkflowProcessor processor) {
    if (_processor == processor) {
      _processor = null;
    }
  }
}
