/*******************************************************************************
 * Copyright (c) 2008, 2009 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation
 *******************************************************************************/

package org.eclipse.smila.utils.xml.stax;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Namespace;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.apache.commons.io.IOUtils;

/**
 * StAX based XML splitter, that parses a XML stream and splits it into multiple XML snippets described by begin and end
 * tags.
 */
public class XmlSnippetSplitter {

  /**
   * The XMLInputFactory.
   */
  private final XMLInputFactory _inputFactory = XMLInputFactory.newInstance();

  /**
   * The XMLOutputFactory.
   */
  private final XMLOutputFactory _outputFactory = XMLOutputFactory.newInstance();

  /**
   * Callback to the XmlSnippetHandler.
   */
  private XmlSnippetHandler _snippetHandler;

  /**
   * The begin of a snippet.
   */
  private QName _begin;

  /**
   * The end of a snippet.
   */
  private QName _end;

  /**
   * Conversion Constructor.
   * 
   * @param snippetHandlet
   *          the XmlSnippetHandler to handle splitted xml snippets
   * @param begin
   *          the MarkerTag that describes the begin of a snippet
   * @param end
   *          the MarkerTag that describes the end of a snippet
   */
  public XmlSnippetSplitter(final XmlSnippetHandler snippetHandlet, final QName begin, final QName end) {
    // check parameters
    if (snippetHandlet == null) {
      throw new IllegalArgumentException("parameter snippetHandler is null");
    }
    if (begin == null) {
      throw new IllegalArgumentException("parameter begin is null");
    }
    if (end == null) {
      throw new IllegalArgumentException("parameter end is null");
    }

    _snippetHandler = snippetHandlet;
    _begin = begin;
    _end = end;
  }

  /**
   * Reads in the given InputStream and parses it for XML snippets describe by the begin and end marker tags. Closes the
   * inputStream when finished.
   * 
   * @param inputStream
   *          the InputStream to read from
   * @throws XMLStreamException
   *           the StAX Exception
   */
  public void read(final InputStream inputStream) throws XMLStreamException {
    if (inputStream != null) {
      try {
        final XMLEventReader eventReader = _inputFactory.createXMLEventReader(inputStream);
        parse(eventReader);
      } finally {
        IOUtils.closeQuietly(inputStream);
      }
    } // if
  }

  /**
   * Parse for xml snippets. If a snippet is found the registered XmlSnippethandler is called.
   * 
   * @param eventReader
   *          the XMLEventReader
   * @throws XMLStreamException
   *           StAX error.
   */
  private void parse(final XMLEventReader eventReader) throws XMLStreamException {
    ByteArrayOutputStream outputStream = null;
    XMLEventWriter eventWriter = null;
    try {
      final List<StartElement> parents = new ArrayList<StartElement>();
      while (eventReader.hasNext()) {
        final XMLEvent event = eventReader.nextEvent();
        if (isSnippetBegin(event)) {
          // begin of snippet
          outputStream = new ByteArrayOutputStream();
          // Use UTF-8 encoding, as our snippets have no XML header that represent the encoding
          eventWriter = _outputFactory.createXMLEventWriter(outputStream, "UTF-8");
          eventWriter.add(event);

          // write all parent namespaces
          writeNamespaces(event.asStartElement(), parents, eventWriter);
        } else if (eventWriter != null) {
          if (isSnippetEnd(event)) {
            // end of snippet
            eventWriter.add(event);
            eventWriter.close();
            _snippetHandler.handleSnippet(outputStream.toByteArray());

            // reset eventWriter and outputStream
            eventWriter = null;
            outputStream = null;
          } else {
            eventWriter.add(event);
          }
        }

        // Push/pop current element
        if (event.isStartElement()) {
          parents.add(event.asStartElement());
        } else if (event.isEndElement()) {
          parents.remove(parents.size() - 1);
        }
      } // while
    } finally {
      if (eventWriter != null) {
        eventWriter.close();
      }
    }
  }

  /**
   * Writes the declared namespaces of all parent elements that are not defined for this element.
   * 
   * @param element
   *          the current element
   * @param parents
   *          the list of all parent elements
   * @param eventWriter
   *          the current writer
   * @throws XMLStreamException
   *           if the stream throws one
   */
  private void writeNamespaces(StartElement element, List<StartElement> parents, XMLEventWriter eventWriter)
    throws XMLStreamException {
    Map<String, Namespace> prefixes = null;
    // Find all namespace prefixes defined in the parent elements
    for (final StartElement parent : parents) {
      for (final Iterator<?> namespaces = parent.getNamespaces(); namespaces.hasNext();) {
        if (prefixes == null) {
          prefixes = new HashMap<String, Namespace>();
        }
        final Namespace namespace = (Namespace) namespaces.next();
        prefixes.put(namespace.getPrefix(), namespace);
      }
    }

    if (prefixes != null) {
      // Remove all prefixes defined for our element
      for (final Iterator<?> namespaces = element.getNamespaces(); namespaces.hasNext();) {
        prefixes.remove(((Namespace) namespaces.next()).getPrefix());
      }

      // Write the remaining namespaces
      for (final Namespace namespace : prefixes.values()) {
        eventWriter.add(namespace);
      }
    }
  }

  /**
   * Checks if the current tag is the begin of a snippet.
   * 
   * @param event
   *          the XMLEvent
   * @return true if the tag name and the start/end tag settings of the _begin MarkerTag match the current events
   *         properties.
   */
  private boolean isSnippetBegin(final XMLEvent event) {
    if (!event.isStartElement()) {
      return false;
    }
    final QName name = event.asStartElement().getName();
    final String ns = _begin.getNamespaceURI();
    return _begin.getLocalPart().equals(name.getLocalPart()) && (ns.isEmpty() || ns.equals(name.getNamespaceURI()));
  }

  /**
   * Checks if the current tag is the end of a snippet.
   * 
   * @param event
   *          the XMLEvent
   * @return true if the tag name and the start/end tag settings of the _end MarkerTag match the current events
   *         properties.
   */
  private boolean isSnippetEnd(final XMLEvent event) {
    if (!event.isEndElement()) {
      return false;
    }
    final QName name = event.asEndElement().getName();
    final String ns = _end.getNamespaceURI();
    return _end.getLocalPart().equals(name.getLocalPart()) && (ns.isEmpty() || ns.equals(name.getNamespaceURI()));
  }
}
