/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.test;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletionService;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.bulkbuilder.BulkbuilderConstants;
import org.eclipse.smila.bulkbuilder.internal.BulkbuilderServiceImpl;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.jobmanager.WorkflowRunInfo;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.input.RecordInput;

/**
 * Testcases for bulkbuilder service.
 * 
 */
public class TestBulkbuilder extends BulkbuilderTestBase {

  /** Logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * Tests if the service can be accessed.
   * 
   * @throws Exception
   *           test fails.
   */
  public void testService() throws Exception {
    _log.info("testService start.");
    assertTrue(_builder instanceof BulkbuilderServiceImpl);
    _log.info("testService end.");
  }

  /**
   * test creation of an add bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddBulk() throws Exception {
    _log.info("testAddBulk start.");
    final String jobName = "testAddBulk";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 5;
    final WorkflowRunInfo firstInfo = _builder.addRecord(jobName, createRecord(0));
    for (int i = 1; i < noOfRecords; i++) {
      final WorkflowRunInfo newInfo = _builder.addRecord(jobName, createRecord(i));
      assertEquals(firstInfo, newInfo);
    }
    final WorkflowRunInfo commitInfo = _builder.commitJob(jobName);
    assertEquals(firstInfo, commitInfo);
    assertNoTask(WORKER_DELETESCONSUMER);
    waitForFinishingTasksProcessed();
    final Task task = getTask(WORKER_BULKCONSUMER);
    final RecordInput reader = readBulk(task, "bulk");
    try {
      final Set<Integer> idSet = new HashSet<Integer>();
      for (int i = 0; i < noOfRecords; i++) {
        final Record record = reader.getRecord();
        assertNotNull(record);
        idSet.add(getRecordIdAsInteger(record));
      }
      assertEquals(noOfRecords, idSet.size());
      finishTaskWithSuccess(task);
      assertNoTask(WORKER_BULKCONSUMER);
      _jobRunEngine.finishJob(jobName, jobId);
      checkCounters(jobName, jobId, 1, noOfRecords, 0, 0);
      _log.info("testAddBulk end.");
      waitForRunningWorker();
    } finally {
      reader.close();
    }
  }

  /**
   * test creation of an add bulk only. Check if attachments have been stored.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddBulkWithAttachment() throws Exception {
    _log.info("testAddBulkWithAttachment start.");
    final String jobName = "testAddBulkWithAttachment";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 5;
    Record record = createRecord(0);
    record.setAttachment("attachment", new byte[0]);
    final WorkflowRunInfo firstInfo = _builder.addRecord(jobName, record);
    for (byte i = 1; i < noOfRecords; i++) {
      record = createRecord(i);
      final byte[] att = new byte[i];
      Arrays.fill(att, i);
      record.setAttachment("attachment", att);
      final WorkflowRunInfo newInfo = _builder.addRecord(jobName, record);
      assertEquals(firstInfo, newInfo);
    }
    final WorkflowRunInfo commitInfo = _builder.commitJob(jobName);
    assertEquals(firstInfo, commitInfo);
    assertNoTask(WORKER_DELETESCONSUMER);
    waitForFinishingTasksProcessed();
    final Task task = getTask(WORKER_BULKCONSUMER);
    final RecordInput reader = readBulk(task, "bulk");
    try {
      final Set<Integer> idSet = new HashSet<Integer>();
      for (int i = 0; i < noOfRecords; i++) {
        final Record checkRecord = reader.getRecord();
        assertNotNull(checkRecord);
        idSet.add(getRecordIdAsInteger(checkRecord));
        final byte[] attachment = checkRecord.getAttachmentAsBytes("attachment");
        assertNotNull(attachment);
        assertEquals(getRecordIdAsInteger(checkRecord).intValue(), attachment.length);
        final byte content = getRecordIdAsInteger(checkRecord).byteValue();
        if (content > 0) {
          assertEquals(content, attachment[0]);
        }
      }
      assertEquals(noOfRecords, idSet.size());
      finishTaskWithSuccess(task);
      assertNoTask(WORKER_BULKCONSUMER);
      _jobRunEngine.finishJob(jobName, jobId);
      checkCounters(jobName, jobId, 1, noOfRecords, 0, 0);
      _log.info("testAddBulkWithAttachment end.");
      waitForRunningWorker();
    } finally {
      reader.close();
    }
  }

  /**
   * test for adding micro bulks. With the values below we should have just one task with a bulk, independent from the
   * number of micro bulks (2).
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddMicroBulk() throws Exception {
    _log.info("testAddMicroBulkIncremental start.");
    final String jobName = "testAddMicroBulkIncremental";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 3;
    final int numberOfMicroBulks = 2;
    for (int j = 0; j < numberOfMicroBulks; j++) {
      final String microBulkId = "08micro15modeIncremental" + j;
      for (int i = 0; i < noOfRecords; i++) {
        _builder.addToMicroBulk(jobName, createRecord(i), microBulkId);
      }
      _builder.finishMicroBulk(jobName, microBulkId);
    }
    _builder.commitJob(jobName);
    assertNoTask(WORKER_DELETESCONSUMER);
    waitForFinishingTasksProcessed();
    final Task task = getTask(WORKER_BULKCONSUMER);
    final RecordInput reader = readBulk(task, "bulk");
    try {
      int numberOfResultRecords = 0;
      final int numberOfAllRecords = numberOfMicroBulks * noOfRecords;
      final Set<Integer> idSet = new HashSet<Integer>();
      for (int i = 0; i < numberOfAllRecords; i++) {
        final Record record = reader.getRecord();
        assertNotNull(record);
        idSet.add(getRecordIdAsInteger(record));
        numberOfResultRecords++;
      }
      assertEquals(noOfRecords, idSet.size());
      assertEquals(numberOfAllRecords, numberOfResultRecords);
      finishTaskWithSuccess(task);
      assertNoTask(WORKER_BULKCONSUMER);
      _jobRunEngine.finishJob(jobName, jobId);
      checkCounters(jobName, jobId, 1, numberOfAllRecords, 0, 0);
      _log.info("testAddMicroBulkIncremental end.");
    } finally {
      reader.close();
    }
  }

  /**
   * test for adding micro bulks. With the values below we should have just one task with a bulk, independent from the
   * number of micro bulks (2).
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddMicroBulkWithAttachments() throws Exception {
    _log.info("testAddMicroBulkWithAttachments start.");
    final String jobName = "testAddMicroBulkWithAttachments";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 3;
    final int numberOfMicroBulks = 2;
    for (int j = 0; j < numberOfMicroBulks; j++) {
      final String microBulkId = "08micro15modeIncremental" + j;
      for (int i = 0; i < noOfRecords; i++) {
        final int id = j * noOfRecords + i;
        final Record record = createRecord(id);
        final byte[] attachment = new byte[id];
        Arrays.fill(attachment, Integer.valueOf(id).byteValue());
        record.setAttachment("attachment", attachment);
        _builder.addToMicroBulk(jobName, record, microBulkId);
      }
      _builder.finishMicroBulk(jobName, microBulkId);
    }
    _builder.commitJob(jobName);
    assertNoTask(WORKER_DELETESCONSUMER);
    waitForFinishingTasksProcessed();
    final Task task = getTask(WORKER_BULKCONSUMER);
    final RecordInput reader = readBulk(task, "bulk");
    try {
      int numberOfResultRecords = 0;
      final int numberOfAllRecords = numberOfMicroBulks * noOfRecords;
      final Set<Integer> idSet = new HashSet<Integer>();
      for (int i = 0; i < numberOfAllRecords; i++) {
        final Record record = reader.getRecord();
        assertNotNull(record);
        idSet.add(getRecordIdAsInteger(record));
        numberOfResultRecords++;
        final byte[] attachment = record.getAttachmentAsBytes("attachment");
        assertNotNull(attachment);
        assertEquals(getRecordIdAsInteger(record).intValue(), attachment.length);
        final byte content = getRecordIdAsInteger(record).byteValue();
        if (content > 0) {
          assertEquals(content, attachment[0]);
        }
      }
      assertEquals(numberOfAllRecords, idSet.size());
      assertEquals(numberOfAllRecords, numberOfResultRecords);
      finishTaskWithSuccess(task);
      assertNoTask(WORKER_BULKCONSUMER);
      _jobRunEngine.finishJob(jobName, jobId);
      checkCounters(jobName, jobId, 1, numberOfAllRecords, 0, 0);
      _log.info("testAddMicroBulkWithAttachments end.");
    } finally {
      reader.close();
    }
  }

  /**
   * test for adding micro bulks where each micro bulk is bigger than limitIncSize.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddMicroBulkBySize() throws Exception {
    _log.info("testAddMicroBulkIncrementalSize start.");
    final String jobName = "testAddMicroBulkIncrementalSize";
    final String jobId = startJob(jobName, "tempstore");
    // 10 records are bigger than limitIncSize=10k
    final int noOfRecords = 10;
    final int numberOfMicroBulks = 2;
    for (int j = 0; j < numberOfMicroBulks; j++) {
      final String microBulkId = "08micro15modeIncrementalSize" + j;
      for (int i = 0; i < noOfRecords; i++) {
        _builder.addToMicroBulk(jobName, createRecord(i), microBulkId);
      }
      _builder.finishMicroBulk(jobName, microBulkId);
    }
    _builder.commitJob(jobName);
    waitForFinishingTasksProcessed();
    Task task = null;
    int numberOfResultRecords = 0;
    int taskCount = 0;
    final int numberOfAllRecords = numberOfMicroBulks * noOfRecords;
    final Set<Integer> idSet = new HashSet<Integer>();
    do {
      task = _taskManager.getTask(WORKER_BULKCONSUMER, null);
      assertNoTask(WORKER_DELETESCONSUMER);
      if (task != null) {
        taskCount++;
        final RecordInput reader = readBulk(task, "bulk");
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              numberOfResultRecords++;
              idSet.add(getRecordIdAsInteger(record));
            }
          } while (record != null);
        } finally {
          reader.close();
        }
        finishTaskWithSuccess(task);
      }
    } while (task != null);
    assertEquals(noOfRecords, idSet.size());
    assertEquals(numberOfAllRecords, numberOfResultRecords);
    assertTrue(taskCount > 1);
    assertNoTask(WORKER_BULKCONSUMER);
    checkCounters(jobName, jobId, taskCount, numberOfAllRecords, 0, 0);
    _jobRunEngine.finishJob(jobName, jobId);
    _log.info("testAddMicroBulkIncrementalSize end.");
  }

  /**
   * test for adding micro bulks with wait to exceed limitIncTime.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddMicroBulkByTime() throws Exception {
    _log.info("testAddMicroBulkIncrementalTime start.");
    final String jobName = "testAddMicroBulkIncrementalTime";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 2;
    final int numberOfMicroBulks = 2;
    for (int j = 0; j < numberOfMicroBulks; j++) {
      final String microBulkId = "08micro15modeIncrementalTime" + j;
      for (int i = 0; i < noOfRecords; i++) {
        _builder.addToMicroBulk(jobName, createRecord(j * noOfRecords + i), microBulkId);
      }
      _builder.finishMicroBulk(jobName, microBulkId);
    }
    Task task = null;
    int numberOfResultRecords = 0;
    int taskCount = 0;
    final int numberOfAllRecords = numberOfMicroBulks * noOfRecords;
    task = waitForTask(WORKER_BULKCONSUMER, 10);
    assertNoTask(WORKER_DELETESCONSUMER);
    taskCount++;
    final RecordInput reader = readBulk(task, "bulk");
    try {
      Record record = null;
      final Set<Integer> idSet = new HashSet<Integer>();
      do {
        record = reader.getRecord();
        if (record != null) {
          numberOfResultRecords++;
          idSet.add(getRecordIdAsInteger(record));
        }
      } while (record != null);
      assertEquals(numberOfMicroBulks * noOfRecords, idSet.size());
      finishTaskWithSuccess(task);
      assertEquals(numberOfAllRecords, numberOfResultRecords);
      assertNoTask(WORKER_BULKCONSUMER);
      assertNoTask(WORKER_DELETESCONSUMER);
      checkCounters(jobName, jobId, taskCount, numberOfAllRecords, 0, 0);
      _jobRunEngine.finishJob(jobName, jobId);
      _log.info("testAddMicroBulkIncrementalTime end.");
    } finally {
      reader.close();
    }
  }

  /**
   * test creation of a delete bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testDeleteBulk() throws Exception {
    _log.info("testDeleteBulk start.");
    final String jobName = "testDeleteBulk";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 5;
    final WorkflowRunInfo firstInfo = _builder.deleteRecord(jobName, createRecord(0));
    for (int i = 1; i < noOfRecords; i++) {
      final WorkflowRunInfo newInfo = _builder.deleteRecord(jobName, createRecord(i));
      assertEquals(firstInfo, newInfo);
    }
    final WorkflowRunInfo commitInfo = _builder.commitJob(jobName);
    assertEquals(firstInfo, commitInfo);
    assertNoTask(WORKER_BULKCONSUMER);
    waitForFinishingTasksProcessed();
    final Task task = getTask(WORKER_DELETESCONSUMER);
    final RecordInput reader = readBulk(task, "deletes");
    try {
      final Set<Integer> idSet = new HashSet<Integer>();
      for (int i = 0; i < noOfRecords; i++) {
        final Record record = reader.getRecord();
        assertNotNull(record);
        idSet.add(getRecordIdAsInteger(record));
      }
      assertEquals(noOfRecords, idSet.size());
      finishTaskWithSuccess(task);
      assertNoTask(WORKER_DELETESCONSUMER);
      _jobRunEngine.finishJob(jobName, jobId);
      checkCounters(jobName, jobId, 0, 0, 1, noOfRecords);
      _log.info("testDeleteBulk end.");
    } finally {
      reader.close();
    }
  }

  /**
   * test creation of an add and delete bulk.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddAndDeleteBulk() throws Exception {
    _log.info("testAddAndDeleteBulk start.");
    final String jobName = "testAddAndDeleteBulk";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 5;
    final WorkflowRunInfo firstInfo = _builder.addRecord(jobName, createRecord(0));
    for (int i = 1; i < noOfRecords; i++) {
      final WorkflowRunInfo newInfo = _builder.addRecord(jobName, createRecord(i));
      assertEquals(firstInfo, newInfo);
    }
    for (int i = 0; i < noOfRecords; i++) {
      final WorkflowRunInfo newInfo = _builder.deleteRecord(jobName, createRecord(noOfRecords + i));
      assertEquals(firstInfo, newInfo);
    }
    final WorkflowRunInfo commitInfo = _builder.commitJob(jobName);
    assertEquals(firstInfo, commitInfo);
    waitForFinishingTasksProcessed();
    final Task addTask = getTask(WORKER_BULKCONSUMER);
    final RecordInput addReader = readBulk(addTask, "bulk");
    try {
      final Set<Integer> idSetAdds = new HashSet<Integer>();
      for (int i = 0; i < noOfRecords; i++) {
        final Record record = addReader.getRecord();
        assertNotNull(record);
        idSetAdds.add(getRecordIdAsInteger(record));
      }
      assertEquals(noOfRecords, idSetAdds.size());
      finishTaskWithSuccess(addTask);
    } finally {
      addReader.close();
    }
    assertNoTask(WORKER_BULKCONSUMER);
    final Task deleteTask = getTask(WORKER_DELETESCONSUMER);
    final RecordInput deleteReader = readBulk(deleteTask, "deletes");
    try {
      final Set<Integer> idSetDeletes = new HashSet<Integer>();
      for (int i = 0; i < noOfRecords; i++) {
        final Record record = deleteReader.getRecord();
        assertNotNull(record);
        idSetDeletes.add(getRecordIdAsInteger(record));
      }
      assertEquals(noOfRecords, idSetDeletes.size());
      finishTaskWithSuccess(deleteTask);
    } finally {
      deleteReader.close();
    }
    assertNoTask(WORKER_DELETESCONSUMER);
    _jobRunEngine.finishJob(jobName, jobId);
    checkCounters(jobName, jobId, 1, noOfRecords, 1, noOfRecords);
    _log.info("testAddAndDeleteBulk end.");
  }

  /**
   * test creation and time-based commit of an add bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddBulkByTime() throws Exception {
    _log.info("testAddBulkByTime start.");
    final String jobName = "testAddBulkByTime";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 10;
    for (int i = 0; i < noOfRecords; i++) {
      _builder.addRecord(jobName, createRecord(i));
      Thread.sleep(ONE_SECOND);
    }
    waitForFinishingTasksProcessed();
    int taskCount = 0;
    int recordCount = 0;
    Task task = null;
    final Set<Integer> idSet = new HashSet<Integer>();
    do {
      final int maxWaitTime = 10;
      task = waitForTaskWithoutAssertion(WORKER_BULKCONSUMER, maxWaitTime);
      if (task != null) {
        taskCount++;
        final RecordInput reader = readBulk(task, "bulk");
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              recordCount++;
              idSet.add(getRecordIdAsInteger(record));
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
      }
    } while (task != null);
    assertEquals(noOfRecords, idSet.size());
    assertNoTask(WORKER_BULKCONSUMER);
    assertNoTask(WORKER_DELETESCONSUMER);
    assertEquals(noOfRecords, recordCount);
    // 10 records pushed over 10 seconds time should go to more than one bulk
    assertTrue(taskCount > 1);
    checkCounters(jobName, jobId, taskCount, noOfRecords, 0, 0);
    _jobRunEngine.finishJob(jobName, jobId);
    _log.info("testAddBulkByTime end.");
  }

  /**
   * test creation and time-based commit of an add bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddBulkByTimeParameter() throws Exception {
    final String jobName = "testAddBulkByTimeParameter";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(BulkbuilderConstants.TASK_PARAM_LIMIT_TIME, 0);
    final String jobId = startJob(jobName, "tempstore", parameters);
    addRecordsAndCheckBulks(jobName, jobId, 5, false, 10, 1);
  }

  /**
   * test creation and job-finish-based commit of an add bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddBulkByFinishJob() throws Exception {
    _log.info("testAddBulkByFinishJob start.");
    final String jobName = "testAddBulkByFinishJob";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 10;
    for (int i = 0; i < noOfRecords; i++) {
      _builder.addRecord(jobName, createRecord(i));
      Thread.sleep(ONE_SECOND);
    }
    // finish triggers the bulk-creation
    _jobRunEngine.finishJob(jobName, jobId);
    waitForFinishingTasksProcessed();
    int taskCount = 0;
    int recordCount = 0;
    Task task = null;
    final Set<Integer> idSet = new HashSet<Integer>();
    do {
      // there will be at least one task but without further delay.
      task = _taskManager.getTask(WORKER_BULKCONSUMER, null);
      if (task != null) {
        taskCount++;
        final RecordInput reader = readBulk(task, "bulk");
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              recordCount++;
              idSet.add(getRecordIdAsInteger(record));
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
      }
    } while (task != null);
    assertEquals(noOfRecords, idSet.size());
    assertNoTask(WORKER_BULKCONSUMER);
    assertNoTask(WORKER_DELETESCONSUMER);
    assertEquals(noOfRecords, recordCount);
    checkCounters(jobName, jobId, taskCount, noOfRecords, 0, 0);
    _log.info("testAddBulkByFinishJob end.");
  }

  /**
   * test creation and time-based commit of a delete bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testDeleteBulkByTime() throws Exception {
    _log.info("testDeleteBulkByTime start.");
    final String jobName = "testDeleteBulkByTime";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 10;
    final Set<Integer> idSet = new HashSet<Integer>();
    for (int i = 0; i < noOfRecords; i++) {
      _builder.deleteRecord(jobName, createRecord(i));
      Thread.sleep(ONE_SECOND);
    }
    waitForFinishingTasksProcessed();
    int taskCount = 0;
    int recordCount = 0;
    Task task = null;
    do {
      final int maxWaitTime = 10;
      task = waitForTaskWithoutAssertion(WORKER_DELETESCONSUMER, maxWaitTime);
      if (task != null) {
        taskCount++;
        final RecordInput reader = readBulk(task, "deletes");
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              recordCount++;
              idSet.add(getRecordIdAsInteger(record));
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
      }
    } while (task != null);
    assertEquals(noOfRecords, idSet.size());
    assertNoTask(WORKER_BULKCONSUMER);
    assertNoTask(WORKER_DELETESCONSUMER);
    assertEquals(noOfRecords, recordCount);
    // 10 records pushed over 10 seconds time should go to more than one bulk
    assertTrue(taskCount > 1);
    _jobRunEngine.finishJob(jobName, jobId);
    checkCounters(jobName, jobId, 0, 0, taskCount, noOfRecords);
    _log.info("testDeleteBulkByTime end.");
  }

  /**
   * test creation and time-based commit of a delete bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testDeleteBulkByTimeParameter() throws Exception {
    final String jobName = "testDeleteBulkByTimeParameter";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(BulkbuilderConstants.TASK_PARAM_LIMIT_TIME, 0);
    final String jobId = startJob(jobName, "tempstore", parameters);
    deleteRecordsAndCheckBulks(jobName, jobId, 5, false, 10, 1);
  }

  /**
   * test creation and job-finish-based commit of an add bulk only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testDeleteBulkByFinishJob() throws Exception {
    _log.info("testDeleteBulkByFinishJob start.");
    final String jobName = "testDeleteBulkByFinishJob";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 10;
    for (int i = 0; i < noOfRecords; i++) {
      _builder.deleteRecord(jobName, createRecord(i));
      Thread.sleep(ONE_SECOND);
    }
    // finish triggers the bulk-creation
    _jobRunEngine.finishJob(jobName, jobId);
    waitForFinishingTasksProcessed();
    int taskCount = 0;
    int recordCount = 0;
    Task task = null;
    final Set<Integer> idSet = new HashSet<Integer>();
    do {
      // there will be at least one task but without further delay.
      task = _taskManager.getTask(WORKER_DELETESCONSUMER, null);
      if (task != null) {
        taskCount++;
        final RecordInput reader = readBulk(task, "deletes");
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              recordCount++;
              idSet.add(getRecordIdAsInteger(record));
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
      }
    } while (task != null);
    assertEquals(noOfRecords, idSet.size());
    assertNoTask(WORKER_BULKCONSUMER);
    assertNoTask(WORKER_DELETESCONSUMER);
    assertEquals(noOfRecords, recordCount);
    checkCounters(jobName, jobId, 0, 0, taskCount, noOfRecords);
    _log.info("testDeleteBulkByFinishJob end.");
  }

  /**
   * test creation and size-based commit of an add bulks only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddBulkBySize() throws Exception {
    _log.info("testAddBulkBySize start.");
    final String jobName = "testAddBulkBySize";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 50;
    final Set<Integer> idSet = new HashSet<Integer>();
    for (int i = 0; i < noOfRecords; i++) {
      _builder.addRecord(jobName, createRecord(i));
    }
    _builder.commitJob(jobName);
    waitForFinishingTasksProcessed();
    int taskCount = 0;
    int recordCount = 0;
    assertNoTask(WORKER_DELETESCONSUMER);
    Task task = null;
    do {
      task = _taskManager.getTask(WORKER_BULKCONSUMER, null);
      if (task != null) {
        taskCount++;
        final RecordInput reader = readBulk(task, "bulk");
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              recordCount++;
              idSet.add(getRecordIdAsInteger(record));
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
      }
    } while (task != null);
    assertEquals(noOfRecords, idSet.size());
    assertNoTask(WORKER_BULKCONSUMER);
    assertEquals(noOfRecords, recordCount);
    checkCounters(jobName, jobId, taskCount, noOfRecords, 0, 0);
    _jobRunEngine.finishJob(jobName, jobId);
    _log.info("testAddBulkBySize end.");
  }

  /**
   * test setting size limit by job parameter.
   */
  public void testAddBulkBySizeParameter() throws Exception {
    final String jobName = "testAddBulkBySizeParameter";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(BulkbuilderConstants.TASK_PARAM_LIMIT_SIZE, "1k");
    // be sure that the timing contraints do not get the bulks committed before...
    parameters.put(BulkbuilderConstants.TASK_PARAM_LIMIT_TIME, 300);
    final String jobId = startJob(jobName, "tempstore", parameters);
    // limit should be reached for each record.
    addRecordsAndCheckBulks(jobName, jobId, 5, false, 0, 5);
  }

  /**
   * test setting size limit by job parameter.
   */
  public void testAddBulkBySizeParameterBig() throws Exception {
    final String jobName = "testAddBulkBySizeParameterBig";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(BulkbuilderConstants.TASK_PARAM_LIMIT_SIZE, "1m");
    // be sure that the timing contraints do not get the bulks committed before...
    parameters.put(BulkbuilderConstants.TASK_PARAM_LIMIT_TIME, 300);
    final String jobId = startJob(jobName, "tempstore", parameters);
    // 50 records, ~1.3kb per record -> 650kb -> only 1 bulk.
    addRecordsAndCheckBulks(jobName, jobId, 500, true, 0, 1);
  }

  /**
   * test creation and size-based commit of an delete bulks only.
   * 
   * @throws Exception
   *           test fails
   */
  public void testDeleteBulkBySize() throws Exception {
    _log.info("testDeleteBulkBySize start.");
    final String jobName = "testDeleteBulkBySize";
    final String jobId = startJob(jobName, "tempstore");
    final int noOfRecords = 50;
    final Set<Integer> idSet = new HashSet<Integer>();
    for (int i = 0; i < noOfRecords; i++) {
      _builder.deleteRecord(jobName, createRecord(i));
    }
    _builder.commitJob(jobName);
    waitForFinishingTasksProcessed();
    int taskCount = 0;
    int recordCount = 0;
    assertNoTask(WORKER_BULKCONSUMER);
    Task task = null;
    do {
      task = _taskManager.getTask(WORKER_DELETESCONSUMER, null);
      if (task != null) {
        taskCount++;
        final RecordInput reader = readBulk(task, "deletes");
        try {
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              recordCount++;
              idSet.add(getRecordIdAsInteger(record));
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
      }
    } while (task != null);
    assertEquals(noOfRecords, idSet.size());
    assertNoTask(WORKER_DELETESCONSUMER);
    assertEquals(noOfRecords, recordCount);
    checkCounters(jobName, jobId, 0, 0, taskCount, noOfRecords);
    _jobRunEngine.finishJob(jobName, jobId);
    _log.info("testDeleteBulkBySize end.");
  }

  /**
   * test setting delbulk size limit by job parameter.
   */
  public void testDeleteBulkBySizeParameter() throws Exception {
    final String jobName = "testDeleteBulkBySizeParameter";
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put(BulkbuilderConstants.TASK_PARAM_LIMIT_SIZE, "100");
    final String jobId = startJob(jobName, "tempstore", parameters);
    deleteRecordsAndCheckBulks(jobName, jobId, 5, false, 0, 5);
  }

  /**
   * test bulk builder with several threads.
   * 
   * @throws Exception
   *           test fails
   */
  public void testAddRecordsMultithreaded() throws Exception {
    _log.info("testAddRecordsMultithreaded start.");
    final String jobNameBase = "testAddRecordsMultithreaded";
    final Map<String, String> jobIds = new HashMap<String, String>();

    final int noOfWorkersPerJob = 5;
    final int noOfJobs = 3;
    final int waitForTermination = 120;
    final ExecutorService executor = Executors.newFixedThreadPool(noOfWorkersPerJob * noOfJobs);
    final CountDownLatch latch = new CountDownLatch(1);
    for (int j = 0; j < noOfJobs; j++) {
      final String jobNameNo = jobNameBase + j;
      jobIds.put(jobNameNo, startJob(jobNameNo, "tempstore"));
      for (int i = 0; i < noOfWorkersPerJob; i++) {
        executor.submit(new DummyRecordAddWorker(false, jobNameNo, _builder,
          i * DummyRecordAddWorker.NO_OF_RECORDS, latch));
      }
    }
    latch.countDown();
    executor.shutdown();
    executor.awaitTermination(waitForTermination, TimeUnit.SECONDS);
    for (final Entry<String, String> jobEntry : jobIds.entrySet()) {
      _jobRunEngine.finishJob(jobEntry.getKey(), jobEntry.getValue());
    }
    waitForFinishingTasksProcessed();
    int recordCount = 0;
    Task task = null;
    final Map<String, Set<Integer>> idSet = new HashMap<String, Set<Integer>>();
    for (final String jobName : jobIds.keySet()) {
      idSet.put(jobName, new HashSet<Integer>());
    }
    do {
      // there will be at least one task but without further delay.
      task = _taskManager.getTask(WORKER_BULKCONSUMER, null);
      if (task != null) {
        final RecordInput reader = readBulk(task, "bulk");
        try {
          final String jobName = task.getProperties().get(Task.PROPERTY_JOB_NAME);
          Record record = null;
          do {
            record = reader.getRecord();
            if (record != null) {
              recordCount++;
              idSet.get(jobName).add(getRecordIdAsInteger(record));
            }
          } while (record != null);
          finishTaskWithSuccess(task);
        } finally {
          reader.close();
        }
      }
    } while (task != null);
    for (final String jobName : jobIds.keySet()) {
      assertEquals(DummyRecordAddWorker.NO_OF_RECORDS * noOfWorkersPerJob, idSet.get(jobName).size());
    }
    assertNoTask(WORKER_BULKCONSUMER);
    assertNoTask(WORKER_DELETESCONSUMER);
    assertEquals(DummyRecordAddWorker.NO_OF_RECORDS * noOfWorkersPerJob * noOfJobs, recordCount);
    _log.info("testAddRecordsMultithreaded end.");
  }

  /**
   * test parallel adding to microbulks and committing. Make sure that we have no deadlock and no Exceptions.
   */
  public void testParallelMicrobulksAndCommit() throws Exception {
    final String jobName = "testParallelMicrobulksAndCommit";
    final String jobId = startJob(jobName, "tempstore");
    final Record recordToAdd = createRecord(0);
    final int noOfThreads = 2;
    final int noOfTries = 100;

    final ExecutorService executor = Executors.newFixedThreadPool(noOfThreads);
    final CompletionService<Void> cs = new ExecutorCompletionService<Void>(executor);

    try {
      // create parallel threads for add/commit
      for (int j = 0; j < noOfThreads; j++) {
        final Callable<Void> callable = new Callable<Void>() {
          private final String _microBulkId = UUID.randomUUID().toString();

          @Override
          public Void call() throws Exception {
            for (int k = 0; k < noOfTries; k++) {
              _builder.addToMicroBulk(jobName, recordToAdd, _microBulkId);
              if (k % 2 == 0) {
                _builder.commitJob(jobName);
              }
              if (k % 5 == 0) {
                _builder.removeMicroBulk(_microBulkId);
              } else {
                _builder.finishMicroBulk(jobName, _microBulkId);
              }
              Thread.sleep(new Random(System.nanoTime()).nextInt(20));
            }
            return null;
          }
        };
        cs.submit(callable);
      }
      // ... but the others should have no errors
      for (int j = 0; j < noOfThreads; j++) {
        cs.take().get();
      }
    } finally {
      executor.shutdownNow();
      _jobRunEngine.cancelJob(jobName, jobId);
    }
  }
}
