/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.httphandler;

import java.net.MalformedURLException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import org.codehaus.jackson.JsonParseException;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.bulkbuilder.InvalidJobException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.validation.InvalidRecordException;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonBulkRequestHandler;
import org.eclipse.smila.jobmanager.WorkflowRunInfo;

/**
 * Implements the processing of HTTP requests for the bulk processing.
 */
public class BulkbuilderBurstHandler extends JsonBulkRequestHandler {

  /** The key for the workflows in the result. */
  public static final String KEY_WORKFLOW_RUNS = "workflowRuns";

  /** The collected workflow run data data to return when calling finish. */
  protected final ThreadLocal<Collection<WorkflowRunInfo>> _workflowRunDataToReturn =
    new ThreadLocal<Collection<WorkflowRunInfo>>();

  /** The micro bulk id. */
  protected final ThreadLocal<String> _microBulkId = new ThreadLocal<String>();

  /** mark if records where actually pushed for a bulk. If not, call commitJob instead of finishMicroBulk in finish */
  protected final Map<String, Boolean> _microBulkHasData = new ConcurrentHashMap<String, Boolean>();

  /** Reference to the BulkBuilder. */
  protected BulkbuilderService _service;

  /**
   * {@inheritDoc}
   */
  @Override
  public void start(final String method, final String requestUri, final HttpExchange exchange) throws Exception {
    _microBulkId.set(UUID.randomUUID().toString());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void process(final String method, final String requestUri, final Record inputRecord,
    final HttpExchange exchange) throws Exception {
    final String jobName = getJobName(requestUri);
    final String microBulkId = _microBulkId.get();
    final WorkflowRunInfo workflowRunInfo = _service.addToMicroBulk(jobName, inputRecord, microBulkId);
    _microBulkHasData.put(microBulkId, Boolean.TRUE);
    appendIntermediateWorkflowRunInfo(workflowRunInfo);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object finish(final String method, final String requestUri, final HttpExchange exchange) throws Exception {
    final String jobName = getJobName(requestUri);
    final String microBulkId = _microBulkId.get();
    final WorkflowRunInfo workflowRunInfo = _service.finishMicroBulk(jobName, microBulkId);
    appendIntermediateWorkflowRunInfo(workflowRunInfo);
    final Collection<WorkflowRunInfo> workflowInfos = _workflowRunDataToReturn.get();
    return createResultObject(workflowInfos);
  }

  /**
   * convert collected WorkflowRunInfos to Any.
   */
  protected Object createResultObject(final Collection<WorkflowRunInfo> workflowInfos) throws MalformedURLException {
    if (workflowInfos != null && !workflowInfos.isEmpty()) {
      final AnyMap result = FACTORY.createAnyMap();
      final AnySeq workflowInfosAny = FACTORY.createAnySeq();
      for (final WorkflowRunInfo workflowInfo : workflowInfos) {
        final Any singleWorkflowInfo =
          BulkbuilderHandler.createResponse(workflowInfo, getRequestHost(), getRootContextPath());
        workflowInfosAny.add(singleWorkflowInfo);
      }
      result.put(KEY_WORKFLOW_RUNS, workflowInfosAny);
      return result;
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void end(final String method, final String requestUri, final HttpExchange exchange) throws Exception {
    final String microBulkId = _microBulkId.get();
    if (microBulkId != null) {
      _service.removeMicroBulk(microBulkId);
      _microBulkHasData.remove(microBulkId);
      _microBulkId.set(null);
    }
    _workflowRunDataToReturn.set(null);
  }

  /**
   * appends the workflow run info to the list if the workflow run id has not already been set.
   * 
   * @param workflowRunInfo
   */
  protected void appendIntermediateWorkflowRunInfo(final WorkflowRunInfo workflowRunInfo) {
    if (workflowRunInfo != null) {
      if (_workflowRunDataToReturn.get() == null) {
        _workflowRunDataToReturn.set(new HashSet<WorkflowRunInfo>());
      }
      _workflowRunDataToReturn.get().add(workflowRunInfo);
    }
  }

  /**
   * Extracts the job name from the requestURI.
   * 
   * @param requestUri
   *          the request URI
   * @return the index name
   */
  protected String getJobName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * {@inheritDoc}
   * 
   * @return HTTP error code matching the exception type.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof InvalidRecordException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex instanceof JsonParseException || ex.getCause() != null && ex.getCause() instanceof JsonParseException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex instanceof InvalidJobException) {
      return HttpStatus.NOT_FOUND;
    }
    if (ex instanceof BulkbuilderException) {
      return HttpStatus.SERVICE_UNAVAILABLE;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * Sets the bulk builder.
   * 
   * @param service
   *          the new BulkBuilder
   */
  public void setBulkbuilderService(final BulkbuilderService service) {
    _service = service;
  }

  /**
   * Unset the bulk builder.
   * 
   * @param service
   *          the BulkBuilder to unset
   */
  public void unsetBulkbuilderService(final BulkbuilderService service) {
    if (_service == service) {
      _service = null;
    }
  }
}
