/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.clusterconfig.simple.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.smila.clusterconfig.ClusterConfigException;
import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.utils.service.ServiceUtils;
import org.junit.Before;
import org.junit.Test;

/**
 * Test cases for {@link SimpleClusterConfigService}.
 */
public class TestSimpleClusterConfig {

  /** the service to test. */
  private ClusterConfigService _service;

  @Before
  public void prepare() throws Exception {
    _service = ServiceUtils.getService(ClusterConfigService.class);
  }

  /**
   * Test access to the service.
   * 
   * @throws InterruptedException
   *           could not fetch service.
   */
  @Test
  public void testService() throws InterruptedException {
    assertNotNull(_service);
    assertTrue(_service instanceof SimpleClusterConfigService);
  }

  /** Test getting all config properties. */
  @Test
  public void testGetAllProperties() throws Exception {
    final AnyMap allProps = _service.getAllProperties();
    assertEquals("node1", allProps.getStringValue("localHost"));
    assertNotNull(allProps.get("clusterNodes"));
    assertEquals(Long.valueOf(90), allProps.getLongValue("zkGcInterval"));
    assertEquals(Long.valueOf(1), allProps.getLongValue("failsafetyLevel"));
    assertNotNull(allProps.get("taskmanager"));
    assertNotNull(allProps.get("workers"));
    assertEquals(Long.valueOf(2), allProps.getMap("workers").getMap("worker2").getLongValue("maxScaleUp"));
    assertNotNull(allProps.get("services"));
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#getClusterNodes()}.
   * 
   * @throws ClusterConfigException
   *           test failed
   */
  @Test
  public void testGetClusterNodes() throws ClusterConfigException {
    final Collection<String> cluster = new ArrayList<String>();
    cluster.add("node1");
    cluster.add("node2");
    assertEquals(cluster, _service.getClusterNodes());
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#getLocalHost()}.
   */
  @Test
  public void testGetLocalHost() {
    assertEquals("node1", _service.getLocalHost());
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#getZkGcInterval()}.
   */
  @Test
  public void testGetZkGcInterval() {
    assertEquals(90, _service.getZkGcInterval());
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#getFailSafetyLevel()}.
   */
  @Test
  public void testGetFailSafetyLevel() {
    assertEquals(1, _service.getFailSafetyLevel());
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#isConfigured()}.
   */
  @Test
  public void testIsConfigured() {
    assertTrue(_service.isConfigured());
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#getMaxScaleUp()}.
   */
  @Test
  public void testGetMaxScaleUp() {
    assertEquals(5, _service.getMaxScaleUp());
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#getMaxRetries()}.
   */
  @Test
  public void testGetMaxRetries() {
    assertEquals(9, _service.getMaxRetries());
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#getTimeToLive()}.
   */
  @Test
  public void testGetTimeToLive() {
    assertEquals(100, _service.getTimeToLive());
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#isResumeJobs()}.
   */
  @Test
  public void testIsResumeJobs() {
    assertEquals(true, _service.isResumeJobs());
  }

  /**
   * Test method for
   * {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#getWorkerScaleUp(java.lang.String)}.
   */
  @Test
  public void testGetWorkerScaleUp() {
    assertEquals(1, _service.getWorkerScaleUp("worker1"));
    assertEquals(2, _service.getWorkerScaleUp("worker2"));
    assertEquals(SimpleClusterConfigService.DEFAULT_MAX_SCALE_UP, _service.getWorkerScaleUp("worker3"));
    assertEquals(SimpleClusterConfigService.DEFAULT_MAX_SCALE_UP, _service.getWorkerScaleUp("worker4"));
  }

  /**
   * Test method for {@link org.eclipse.smila.clusterconfig.simple.SimpleClusterConfigService#getWorkersWithScaleUp()}.
   */
  @Test
  public void testGetWorkersWithScaleUp() {
    final Collection<String> scaledUpWorkers = _service.getWorkersWithScaleUp();
    assertEquals(2, scaledUpWorkers.size());
    assertTrue(scaledUpWorkers.contains("worker1"));
    assertTrue(scaledUpWorkers.contains("worker2"));
  }

  /** tests {@link SimpleClusterConfigService#getHttpPort(String)}. */
  @Test
  public void testGetHttpPort() {
    int httpPort = _service.getHttpPort("smila");
    assertEquals(8080, httpPort);
    httpPort = _service.getHttpPort("non existing service");
    assertEquals(-1, httpPort);

  }
}
