/***********************************************************************************************************************
 * Copyright (c) 2008,2011 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the 
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/
package org.eclipse.smila.datamodel.ipc.test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;

/**
 * @author scank01
 * 
 */
public class TestBinaryObjectStreamIterator extends TestCase {

  /** no of Any objects. */
  private static final int NO_OF_ANIES = 100;

  /** content of Any objects. */
  private static final String CONTENT = "Dummy content.";

  /** BON writer. */
  private final IpcAnyWriter _anyWriter = new IpcAnyWriter();

  /** binary stream. */
  private CloseCheckByteArrayInputStream _binaryStream;

  /** binary stream with only one element. */
  private CloseCheckByteArrayInputStream _binarySingleStream;

  /** non-binary stream. */
  private CloseCheckByteArrayInputStream _nonBinaryStream;

  /** store of the any objects. */
  private List<Any> _anyList;

  /** empty stream. */
  private final CloseCheckInputStream _emptyStream = new CloseCheckInputStream();

  /** close call checking input stream. */
  private final class CloseCheckInputStream extends InputStream {
    /** has close been called? */
    private boolean _closeCalled;

    /**
     * {@inheritDoc}
     */
    @Override
    public int read() throws IOException {
      return -1;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void close() throws IOException {
      _closeCalled = true;
    };
  }

  /**
   * test class to check if close() has been called on the stream.
   */
  private final class CloseCheckByteArrayInputStream extends ByteArrayInputStream {

    /** has close been called? */
    private boolean _closeCalled;

    /**
     * Constructor.
     * 
     * @param arg0
     *          the byte array.
     */
    public CloseCheckByteArrayInputStream(final byte[] arg0) {
      super(arg0);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void close() throws IOException {
      super.close();
      _closeCalled = true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _anyList = new ArrayList<Any>();
    final ByteArrayOutputStream binaryOutStream = new ByteArrayOutputStream();
    final ByteArrayOutputStream binarySingleOutStream = new ByteArrayOutputStream();
    final ByteArrayOutputStream nonBinaryOutStream = new ByteArrayOutputStream();
    for (int i = 0; i < NO_OF_ANIES; i++) {
      final AnyMap any = DataFactory.DEFAULT.createAnyMap();
      any.put("id", i);
      any.put("content", CONTENT + i);
      _anyWriter.writeBinaryStream(any, binaryOutStream);
      if (i == 0) {
        _anyWriter.writeBinaryStream(any, binarySingleOutStream);
      }
      _anyWriter.writeJsonStream(any, nonBinaryOutStream);
      _anyList.add(any);
    }
    _binaryStream = new CloseCheckByteArrayInputStream(binaryOutStream.toByteArray());
    _nonBinaryStream = new CloseCheckByteArrayInputStream(nonBinaryOutStream.toByteArray());
    _binarySingleStream = new CloseCheckByteArrayInputStream(binarySingleOutStream.toByteArray());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator#hasNext()}.
   * 
   * @throws IOException
   *           test failed
   */
  public final void testHasNext() throws IOException {
    final BinaryObjectStreamIterator iter = new BinaryObjectStreamIterator(_emptyStream);
    assertFalse(iter.hasNext());
    assertTrue(_emptyStream._closeCalled);

    final BinaryObjectStreamIterator iterBin = new BinaryObjectStreamIterator(_binaryStream);
    assertTrue(iterBin.hasNext());
    assertFalse(_binaryStream._closeCalled);

    final BinaryObjectStreamIterator iterSingleBin = new BinaryObjectStreamIterator(_binarySingleStream);
    assertTrue(iterSingleBin.hasNext());
    assertFalse(_binarySingleStream._closeCalled);

    final BinaryObjectStreamIterator iterNonBin = new BinaryObjectStreamIterator(_nonBinaryStream);
    try {
      iterNonBin.hasNext();
      fail("IllegalStateException expected.");
    } catch (final IllegalStateException e) {
      ;// ok
      assertTrue(_nonBinaryStream._closeCalled);
    }
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator#next()}.
   * 
   * @throws IOException
   *           test failed.
   */
  public final void testNext() throws IOException {
    final BinaryObjectStreamIterator iter = new BinaryObjectStreamIterator(_emptyStream);
    try {
      iter.next();
      fail("NoSuchElementException expected.");
    } catch (final NoSuchElementException ex) {
      ; // ok
      assertTrue(_emptyStream._closeCalled);
    }

    final BinaryObjectStreamIterator iterBin = new BinaryObjectStreamIterator(_binaryStream);
    final List<Any> results = new ArrayList<Any>();
    while (iterBin.hasNext()) {
      results.add(iterBin.next().getMetadata());
    }
    assertEquals(_anyList, results);
    assertTrue(_binaryStream._closeCalled);

    final BinaryObjectStreamIterator iterSingleBin = new BinaryObjectStreamIterator(_binarySingleStream);
    final List<Any> resultsSingle = new ArrayList<Any>();
    while (iterSingleBin.hasNext()) {
      resultsSingle.add(iterSingleBin.next().getMetadata());
    }
    assertEquals(1, resultsSingle.size());
    assertEquals(_anyList.get(0), resultsSingle.get(0));
    assertTrue(_binarySingleStream._closeCalled);

    final BinaryObjectStreamIterator iterNonBin = new BinaryObjectStreamIterator(_nonBinaryStream);
    try {
      iterNonBin.next();
      fail("IllegalStateException expected.");
    } catch (final IllegalStateException ex) {
      ; // ok
      assertTrue(_nonBinaryStream._closeCalled);
    }
  }

  /**
   * tests {@link BinaryObjectStreamIterator#close()}.
   * 
   * @throws IOException
   *           test failed.
   */
  public final void testClose() throws IOException {
    final BinaryObjectStreamIterator iterBin = new BinaryObjectStreamIterator(_binaryStream);
    final List<Any> results = new ArrayList<Any>();
    while (iterBin.hasNext()) {
      results.add(iterBin.next().getMetadata());
      // close pre-maturely.
      iterBin.close();
    }
    assertEquals(1, results.size());
    assertTrue(_binaryStream._closeCalled);
    // can be closed again without any exceptions.
    iterBin.close();
  }

}
