/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.datamodel.test;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.ValueFormatHelper;

/**
 * Test cases for {@link ValueFormatHelper}.
 */
public class TestValueFormatHelper extends TestCase {

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
  }

  /** {@inheritDoc} */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ValueFormatHelper#getDefaultDateTimeFormat()} .
   */
  public void testGetDefaultDateTimeFormat() {
    assertEquals("yyyy-MM-dd'T'HH:mm:ss.SSSZ", ValueFormatHelper.getDefaultDateTimeFormat().toPattern());
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ValueFormatHelper#formatDate(java.util.Date)} .
   */
  public void testFormatDate() {
    final Calendar cal = new GregorianCalendar();
    cal.set(2011, 11, 01);
    final String dateStr = ValueFormatHelper.INSTANCE.formatDate(cal.getTime());
    assertEquals("2011-12-01", dateStr);
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ValueFormatHelper#formatDateTime(java.util.Date)} .
   * 
   * @throws ParseException
   */
  public void testFormatDateTime() throws ParseException {
    final Calendar cal = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
    cal.set(2011, 11, 01, 1, 2, 3);
    cal.set(Calendar.MILLISECOND, 123);
    final String dateStr = ValueFormatHelper.INSTANCE.formatDateTime(cal.getTime());
    final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
    assertEquals(cal.getTime(), sdf.parse(dateStr));
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ValueFormatHelper#parseDate(java.lang.String)} .
   * 
   * @throws ParseException
   */
  public void testParseDate() throws ParseException {
    final Calendar cal = new GregorianCalendar();
    cal.set(2011, 11, 01, 0, 0, 0);
    cal.set(Calendar.MILLISECOND, 0);
    final Date testDate = cal.getTime();
    final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
    assertEquals(testDate, ValueFormatHelper.INSTANCE.parseDate(sdf.format(testDate)));
    final String notADate = "2011-11-1x";
    try {
      ValueFormatHelper.INSTANCE.parseDate(notADate);
      fail("Exception expected.");
    } catch (final ParseException e) {
      ;// ok
    }
    try {
      ValueFormatHelper.INSTANCE.parseDate("2011-02-29");
      fail("Exception expected.");
    } catch (final ParseException e) {
      ;// ok
    }
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ValueFormatHelper#tryParseDate(java.lang.String)} .
   */
  public void testTryParseDate() {
    final Calendar cal = new GregorianCalendar();
    cal.set(2011, 11, 01, 0, 0, 0);
    cal.set(Calendar.MILLISECOND, 0);
    final Date testDate = cal.getTime();
    final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
    assertEquals(testDate, ValueFormatHelper.INSTANCE.tryParseDate(sdf.format(testDate)));
    final String notADate = "2011-11-1x";
    assertNull(ValueFormatHelper.INSTANCE.tryParseDate(notADate));
    assertNull(ValueFormatHelper.INSTANCE.tryParseDate("2011-02-29"));
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ValueFormatHelper#parseDateTime(java.lang.String)} .
   */
  public void testParseDateTime() {
    final String[] stringsToTest =
      { "2011-01-02T00:02:03.000-0100", "2011-01-02T01:02:03.000+0000", "2011-01-02T01:02:03.000-0000",
        "2011-01-02T02:32:03+0130", "2011-01-02T01:02:03.000Z", "2011-01-02T01:02:03Z",
        "abcd-01-02T01:02:03.000+0000", "2011-01-02T01:02:DE+0000", "asdasdasd", "adfjqpfjsfj wqoefij opqw ",
        "asdpasodpaosdjpoasjdpasd", " asdfpoaskp aspokdp a", "2011-01-02T01:02:03O", "epqwokp",
        "dfpasdfkpsdofkapsdofkasdpofkapsdfkasdpfoksd", "fpasdjfpasdfjp", "pdfjfpadfpasd aspdf psdfjap sfjpsdif",
        " apsdfjpasdfj pasdfpjsdp fsdapfisdfiasdpfj asdpfj apsdfjapsdfjspdifjasp fajsdpf asdpfjid " };
    final Calendar cal = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
    cal.set(2011, 0, 2, 1, 2, 3);
    cal.set(Calendar.MILLISECOND, 0);
    final Date refDate = cal.getTime();
    int datesFound = 0;
    for (final String str : stringsToTest) {
      try {
        final Date date = ValueFormatHelper.INSTANCE.parseDateTime(str);
        if (date != null) {
          ++datesFound;
          assertEquals(refDate, date);
        }
      } catch (final ParseException e) {
        ;// ignore
      }
    }
    assertEquals(6, datesFound);
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ValueFormatHelper#tryParseDateTime(java.lang.String)} .
   */
  public void testTryParseDateTime() {
    final String[] stringsToTest =
      { "2011-01-02T00:02:03.000-0100", "2011-01-02T01:02:03.000+0000", "2011-01-02T01:02:03.000-0000",
        "2011-01-02T02:32:03+0130", "2011-01-02T01:02:03.000Z", "2011-01-02T01:02:03Z",
        "abcd-01-02T01:02:03.000+0000", "2011-01-02T01:02:DE+0000", "asdasdasd", "adfjqpfjsfj wqoefij opqw ",
        "asdpasodpaosdjpoasjdpasd", " asdfpoaskp aspokdp a", "2011-01-02T01:02:03O", "epqwokp",
        "dfpasdfkpsdofkapsdofkasdpofkapsdfkasdpfoksd", "fpasdjfpasdfjp", "pdfjfpadfpasd aspdf psdfjap sfjpsdif",
        " apsdfjpasdfj pasdfpjsdp fsdapfisdfiasdpfj asdpfj apsdfjapsdfjspdifjasp fajsdpf asdpfjid " };
    final Calendar cal = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
    cal.set(2011, 0, 2, 1, 2, 3);
    cal.set(Calendar.MILLISECOND, 0);
    final Date refDate = cal.getTime();
    int datesFound = 0;
    for (final String str : stringsToTest) {
      final Date date = ValueFormatHelper.INSTANCE.tryParseDateTime(str);
      if (date != null) {
        ++datesFound;
        assertEquals(refDate, date);
      }
    }
    assertEquals(6, datesFound);
  }

  /**
   * Test method for {@link org.eclipse.smila.datamodel.ValueFormatHelper#tryParseDateTime(java.lang.String)} .
   */
  public void testTryParseISO8601DateTime() {
    final String[] stringsToTest =
      { "2011-01-02T05:15:03Z", "2011-01-02T07:15:03+02", "2011-01-02T07:45:03+0230", "2011-01-02T07:45:03+02:30",
        "2011-01-02T05:15:03.000Z", "2011-01-02T07:15:03.000+02", "2011-01-02T07:45:03.000+0230",
        "2011-01-02T07:45:03.000+02:30" };
    final Calendar cal = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
    // 02.01.2011T05:15:03.000UTC
    cal.set(2011, 0, 2, 5, 15, 3);
    cal.set(Calendar.MILLISECOND, 0);
    final Date refDate = cal.getTime();
    int datesFound = 0;
    for (final String str : stringsToTest) {
      final Date date = ValueFormatHelper.INSTANCE.tryParseDateTime(str);
      if (date != null) {
        ++datesFound;
        assertEquals(refDate, date);
      }
    }
    assertEquals(8, datesFound);
  }
}
