/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.test.httpserver;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jetty.xml.XmlConfiguration;
import org.eclipse.smila.http.server.HttpHandler;
import org.eclipse.smila.http.server.JettyHandler;
import org.eclipse.smila.http.server.internal.HttpServer;
import org.eclipse.smila.http.server.internal.HttpServiceImpl;
import org.eclipse.smila.utils.config.ConfigUtils;

/**
 * multi http server service to check failover.
 */
public class MultiHttpServerServiceImpl implements MultiHttpServerService {
  /** bundle name. */
  private static final String BUNDLE_NAME = "org.eclipse.smila.http.client.test.httpserver";

  /** the configured servers. */
  private final Map<Integer, HttpServer> _server = new HashMap<Integer, HttpServer>();

  /** A list of all registered HttpHandlers. */
  private final Collection<HttpHandler> _handlers = new ArrayList<HttpHandler>();

  /** A list of all registered JettyHandlers. */
  private final Collection<JettyHandler> _jettyHandlers = new ArrayList<JettyHandler>();

  /** local logger. */
  private final Log _log = LogFactory.getLog(HttpServiceImpl.class);

  /**
   * DS deactivate method.
   */
  protected void deactivate() {
    for (final Entry<Integer, HttpServer> serverEntry : _server.entrySet()) {
      try {
        serverEntry.getValue().stop();
      } catch (final Exception e) {
        final String msg = "Stopping HTTP server failed";
        _log.error(msg, e);
      }
    }
  }

  /** {@inheritDoc} */
  @Override
  public void addHttpHandler(final HttpHandler handler) {
    _handlers.add(handler);
  }

  /** {@inheritDoc} */
  @Override
  public void removeHttpHandler(final HttpHandler handler) {
    _handlers.remove(handler);
  }

  /** {@inheritDoc} */
  @Override
  public void addJettyHandler(final JettyHandler handler) {
    _jettyHandlers.add(handler);
  }

  /** {@inheritDoc} */
  @Override
  public void removeJettyHandler(final JettyHandler handler) {
    _jettyHandlers.remove(handler);
  }

  /** {@inheritDoc} */
  @Override
  public void start(final int portNumber) throws Exception {
    final ClassLoader tccl = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(getClass().getClassLoader());
    try {
      if (!_server.containsKey(portNumber)) {
        final InputStream configFileStream = ConfigUtils.getConfigStream(BUNDLE_NAME, "jetty.xml");
        final XmlConfiguration configuration = new XmlConfiguration(configFileStream);
        configuration.getProperties().put("jetty.port", String.valueOf(portNumber));
        final HttpServer server = new HttpServer(configuration);
        for (final HttpHandler handler : _handlers) {
          server.addHttpHandler(handler);
        }
        for (final JettyHandler handler : _jettyHandlers) {
          server.addJettyHandler(handler);
        }
        _server.put(Integer.valueOf(portNumber), server);
      }
      _server.get(portNumber).start();
    } finally {
      Thread.currentThread().setContextClassLoader(tccl);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void stop(final int portNumber) throws Exception {
    if (_server.containsKey(portNumber)) {
      final HttpServer server = _server.remove(portNumber);
      server.stop();
    }
  }
}
