/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky, Andreas Schank (both Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import org.apache.http.HttpEntity;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.ContentBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.params.HttpParams;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.http.client.Attachments;
import org.eclipse.smila.http.client.HttpMethod;
import org.eclipse.smila.http.client.impl.base.HttpRequestFactory;

/**
 * Factory for request related Http client objects.
 * 
 * @since 1.1.0
 */
public class DefaultHttpRequestFactory implements HttpRequestFactory {
  /** Name of the JSON part in multipart requests with attachments. */
  private static final String PARTNAME_RECORD = "_record";

  /** Converter for Anys to JSON. */
  private final IpcAnyWriter _jsonWriter = new IpcAnyWriter(false);

  /* (non-Javadoc)
   * @see org.eclipse.smila.http.client.impl.HttpRequestFactory#getHttpMethod(org.eclipse.smila.http.client.HttpMethod, java.lang.String)
   */
  @Override
  public HttpUriRequest getHttpMethod(final HttpMethod method, final String url) {
    switch (method) {
      case GET:
        return new HttpGet(url);
      case POST:
        return new HttpPost(url);
      case PUT:
        return new HttpPut(url);
      case DELETE:
        return new HttpDelete(url);
      default:
        throw new IllegalArgumentException("Unsupported HTTP method " + method.name());
    }
  }

  /* (non-Javadoc)
   * @see org.eclipse.smila.http.client.impl.HttpRequestFactory#getHttpMethod(org.eclipse.smila.http.client.HttpMethod, java.lang.String, org.apache.http.params.HttpParams)
   */
  @Override
  public HttpUriRequest getHttpMethod(final HttpMethod method, final String url, final HttpParams methodParams) {
    final HttpUriRequest httpMethod = getHttpMethod(method, url);
    if (methodParams != null) {
      httpMethod.setParams(methodParams);
    }
    return httpMethod;
  }

  /* (non-Javadoc)
   * @see org.eclipse.smila.http.client.impl.HttpRequestFactory#createJsonEntity(org.eclipse.smila.datamodel.Any)
   */
  @Override
  public HttpEntity createJsonEntity(final Any parameters) throws IOException {
    if (parameters == null) {
      return null;
    }
    final String jsonParameters = _jsonWriter.writeJsonObject(parameters);
    return new StringEntity(jsonParameters, MIMETYPE_JSON, ENCODING);
  }

  /* (non-Javadoc)
   * @see org.eclipse.smila.http.client.impl.HttpRequestFactory#createJsonEntity(java.io.InputStream)
   */
  @Override
  public HttpEntity createJsonEntity(final InputStream inputStream) throws IOException {
    return createEntity(inputStream, CONTENTTYPE_JSON);
  }

  /* (non-Javadoc)
   * @see org.eclipse.smila.http.client.impl.HttpRequestFactory#createJsonEntity(java.io.InputStream)
   */
  @Override
  public HttpEntity createEntity(final InputStream inputStream, final String contentType) throws IOException {
    if (inputStream == null) {
      return null;
    }
    final InputStreamEntity entity = new InputStreamEntity(inputStream, -1);
    entity.setContentType(contentType);
    return entity;
  }

  /* (non-Javadoc)
   * @see org.eclipse.smila.http.client.impl.HttpRequestFactory#createHttpEntity(org.eclipse.smila.datamodel.Any, org.eclipse.smila.http.client.Attachments)
   */
  @Override
  public HttpEntity createHttpEntity(final Any parameters, final Attachments attachments) throws IOException {
    final HttpEntity requestEntity;
    if (attachments == null) {
      requestEntity = createJsonEntity(parameters);
    } else {
      requestEntity = createMultipartEntity(parameters, attachments);
    }
    return requestEntity;
  }

  /* (non-Javadoc)
   * @see org.eclipse.smila.http.client.impl.HttpRequestFactory#createMultipartEntity(org.eclipse.smila.datamodel.Any, org.eclipse.smila.http.client.Attachments)
   */
  @Override
  public HttpEntity createMultipartEntity(final Any parameters, final Attachments attachments) throws IOException {
    final MultipartEntity requestEntity = new MultipartEntity(HttpMultipartMode.STRICT);
    final String jsonParameters;
    if (parameters == null) {
      jsonParameters = "{}";
    } else {
      jsonParameters = _jsonWriter.writeJsonObject(parameters);
    }
    final StringBody jsonBody = new StringBody(jsonParameters, MIMETYPE_JSON, CHARSET);
    requestEntity.addPart(PARTNAME_RECORD, jsonBody);
    for (final Map.Entry<String, ContentBody> attachmentPart : attachments.getAttachmentParts().entrySet()) {
      requestEntity.addPart(attachmentPart.getKey(), attachmentPart.getValue());
    }
    return requestEntity;
  }

}
