/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file.test;

import java.nio.file.Path;

import junit.framework.TestCase;

import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.file.FileCrawlerService;
import org.eclipse.smila.importing.crawler.file.internal.FileToRecordConverter;
import org.eclipse.smila.importing.util.FilePathNormalizer;

public class TestFileToRecordConverter extends TestCase {

  private final FileToRecordConverter _converter = new FileToRecordConverter(DataFactory.DEFAULT);

  private Path _rootFolder;

  @Override
  public void setUp() throws Exception {
    _rootFolder = FilesystemHelper.initTestFilesystem();
  }

  @Override
  public void tearDown() throws Exception {
    FilesystemHelper.deleteDirectory(_rootFolder);
  }

  public void testFileToRecordWithoutAttachment() throws Exception {
    final Record record = _converter.fileToRecord(_rootFolder.resolve("d.e"), "testDataSource", false);
    assertNotNull(record);
    assertEquals("testDataSource", record.getSource());
    assertEquals("d.e", record.getMetadata().getStringValue(FileCrawlerService.PROPERTY_FILE_NAME));
    assertEquals("e", record.getMetadata().getStringValue(FileCrawlerService.PROPERTY_FILE_EXTENSION));
    assertEquals(FilePathNormalizer.getNormalizedPath(_rootFolder.toString()),
      record.getMetadata().getStringValue(FileCrawlerService.PROPERTY_FILE_FOLDER));
    assertEquals(0, record.getMetadata().getLongValue(FileCrawlerService.PROPERTY_FILE_SIZE).longValue());
    assertTrue(record.getMetadata().containsKey(FileCrawlerService.PROPERTY_FILE_PATH));
    assertTrue(record.getMetadata().containsKey(FileCrawlerService.PROPERTY_FILE_LAST_MODIFIED));
    assertTrue(record.getMetadata().containsKey(ImportingConstants.ATTRIBUTE_DELTA_HASH));
    assertFalse(record.hasAttachment(FileCrawlerService.ATTACHMENT_FILE_CONTENT));
  }

  public void testFileToRecordWithAttachment() throws Exception {
    final Record record = _converter.fileToRecord(_rootFolder.resolve("d.e"), "testDataSource", true);
    assertNotNull(record);
    assertEquals("testDataSource", record.getSource());
    assertEquals("d.e", record.getMetadata().getStringValue(FileCrawlerService.PROPERTY_FILE_NAME));
    assertEquals("e", record.getMetadata().getStringValue(FileCrawlerService.PROPERTY_FILE_EXTENSION));
    assertEquals(FilePathNormalizer.getNormalizedPath(_rootFolder.toString()),
      record.getMetadata().getStringValue(FileCrawlerService.PROPERTY_FILE_FOLDER));
    assertEquals(0, record.getMetadata().getLongValue(FileCrawlerService.PROPERTY_FILE_SIZE).longValue());
    assertTrue(record.getMetadata().containsKey(FileCrawlerService.PROPERTY_FILE_PATH));
    assertTrue(record.getMetadata().containsKey(FileCrawlerService.PROPERTY_FILE_LAST_MODIFIED));
    assertTrue(record.getMetadata().containsKey(ImportingConstants.ATTRIBUTE_DELTA_HASH));
    assertTrue(record.hasAttachment(FileCrawlerService.ATTACHMENT_FILE_CONTENT));
  }

  public void testFileToRecordWithoutExtension() throws Exception {
    final Record record = _converter.fileToRecord(_rootFolder.resolve("c"), "testDataSource", true);
    assertNotNull(record);
    assertEquals("testDataSource", record.getSource());
    assertEquals("c", record.getMetadata().getStringValue(FileCrawlerService.PROPERTY_FILE_NAME));
    assertEquals("", record.getMetadata().getStringValue(FileCrawlerService.PROPERTY_FILE_EXTENSION));
  }

  public void testFileToRecordOnDir() throws Exception {
    try {
      _converter.fileToRecord(_rootFolder.resolve("a"), "testDataSource", false);
      fail("expected IllegalArgumentException");
    } catch (final IllegalArgumentException e) {
      assertNotNull(e);
    }
  }

  public void testDirectoryToRecord() throws Exception {
    final Record record = _converter.dirToRecord(_rootFolder.resolve("a"), "testDataSource");
    assertNotNull(record);
    assertEquals("testDataSource", record.getSource());
    assertTrue(record.getMetadata().containsKey(FileCrawlerService.PROPERTY_FILE_FOLDER));
  }

  public void testDirectoryToRecordOnFile() throws Exception {
    try {
      _converter.dirToRecord(_rootFolder.resolve("c"), "testDataSource");
      fail("expected IllegalArgumentException");
    } catch (final IllegalArgumentException e) {
      assertNotNull(e);
    }
  }

  public void testAddAttachment() throws Exception {
    final Record record = _converter.fileToRecord(_rootFolder.resolve("c"), "testDataSource", false);
    assertNotNull(record);
    assertFalse(record.hasAttachment(FileCrawlerService.ATTACHMENT_FILE_CONTENT));
    _converter.addAttachment(record, FileCrawlerService.PROPERTY_FILE_PATH,
      FileCrawlerService.ATTACHMENT_FILE_CONTENT);
    assertTrue(record.hasAttachment(FileCrawlerService.ATTACHMENT_FILE_CONTENT));
  }

  public void testAddAttachmentWithoutPath() throws Exception {
    final Record record = _converter.fileToRecord(_rootFolder.resolve("c"), "testDataSource", false);
    assertNotNull(record);
    assertFalse(record.hasAttachment(FileCrawlerService.ATTACHMENT_FILE_CONTENT));
    record.getMetadata().remove(FileCrawlerService.PROPERTY_FILE_PATH);
    try {
      _converter.addAttachment(record, FileCrawlerService.PROPERTY_FILE_PATH,
        FileCrawlerService.ATTACHMENT_FILE_CONTENT);
      fail("expected IllegalArgumentException");
    } catch (final IllegalArgumentException e) {
      assertNotNull(e);
    }
  }

}
