/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file;

import java.io.IOException;
import java.nio.file.Paths;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.util.PropertyNameMapper;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.output.RecordOutput;

public class FileFetcherWorker implements Worker {

  public static final String NAME = "fileFetcher";

  public static final String INPUT_SLOT = "filesToFetch";

  public static final String OUTPUT_SLOT = "files";

  private final Log _log = LogFactory.getLog(getClass());

  private FileCrawlerService _fileCrawler;

  @Override
  public String getName() {
    return NAME;
  }

  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    final RecordInput recordInput = taskContext.getInputs().getAsRecordInput(INPUT_SLOT);
    final RecordOutput recordOutput = taskContext.getOutputs().getAsRecordOutput(OUTPUT_SLOT);
    Record record;
    do {
      record = recordInput.getRecord();
      if (record != null) {
        if (_log.isDebugEnabled()) {
          _log.debug("fetching content for record " + record.getId());
        }
        fetch(record, taskContext);
        recordOutput.writeRecord(record);
        if (_log.isDebugEnabled()) {
          _log.debug("added record " + record.getId());
        }
      }
    } while (record != null && !taskContext.isCanceled());
  }

  /** DS service reference bind method. */
  public void setFileCrawlerService(final FileCrawlerService fileCrawler) {
    _fileCrawler = fileCrawler;
  }

  /** DS service reference unbind method. */
  public void unsetFileCrawlerService(final FileCrawlerService fileCrawler) {
    if (_fileCrawler == fileCrawler) {
      _fileCrawler = null;
    }
  }

  /**
   * fetch content and metadata, if record does not already contain them. Measure time for fetching as
   * "duration...fetchContent".
   */
  private void fetch(final Record record, final TaskContext taskContext) throws IOException {
    final PropertyNameMapper mapper = PropertyNameMapper.createFrom(taskContext);
    String pathAttribute = FileCrawlerService.PROPERTY_FILE_PATH;
    if (mapper.containsMapping(FileCrawlerService.PROPERTY_FILE_PATH)) {
      pathAttribute = mapper.get(FileCrawlerService.PROPERTY_FILE_PATH).get(0);
    }
    fillFileContent(mapper, taskContext, record, pathAttribute);
    fillMissingMetadata(mapper, taskContext, record, pathAttribute);
  }

  /** fill content attachments if not set. */
  private void fillFileContent(final PropertyNameMapper mapper, final TaskContext taskContext, final Record record,
    final String pathAttribute) {
    try {
      for (final String attachmentName : mapper.get(FileCrawlerService.ATTACHMENT_FILE_CONTENT)) {
        if (!record.hasAttachment(attachmentName)) {
          final long time = taskContext.getTimestamp();
          _fileCrawler.addAttachment(record, pathAttribute, attachmentName);
          taskContext.measureTime("fetchContent", time);
        }
      }
    } catch (final IOException e) {
      taskContext.getLog().warn("Failed to fetch file for record " + record.getId(), e);
    }
  }

  /** fill metadata properties if not set. */
  private void fillMissingMetadata(final PropertyNameMapper mapper, final TaskContext taskContext,
    final Record record, final String pathAttribute) {
    final AnyMap metadata = record.getMetadata();
    try {
      final Record mdRecord =
        _fileCrawler.fileToRecord(Paths.get(metadata.getStringValue(pathAttribute)), record.getSource(), false);
      final Set<String> propertyNames = _fileCrawler.getFilePropertyNames();
      for (final String property : propertyNames) {
        if (!FileCrawlerService.ATTACHMENT_FILE_CONTENT.equals(property) && mapper.containsMapping(property)) {
          for (final String attributeName : mapper.get(property)) {
            if (!metadata.containsKey(attributeName)) {
              metadata.put(attributeName, mdRecord.getMetadata().get(property));
            }
          }
        }
      }
    } catch (final IOException e) {
      taskContext.getLog().warn("Failed to get metadata from file for record " + record.getId(), e);
    }
  }

}
