/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.jdbc.test;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.jdbc.JdbcCrawlerWorker;
import org.eclipse.smila.importing.crawler.jdbc.JdbcFetcherWorker;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreObject;

/**
 * Base for JDBC crawler tests that use a job for testing.
 */
public abstract class JdbcCrawlerJobTestBase extends JdbcCrawlerTestBase {

  protected static final int DEFAULT_WAIT_TIME = 30000;

  protected static final String STORE = "records";

  protected static final String DATA_SOURCE = "crawlDB";

  protected JobRunEngine _jobRunEngine;

  protected JobRunDataProvider _jobRunDataProvider;

  protected ObjectStoreService _objectStore;

  protected DefinitionPersistence _definitionPersistence;

  @Override
  public void setUp() throws Exception {
    super.setUp();
    _jobRunEngine = getService(JobRunEngine.class);
    _jobRunDataProvider = getService(JobRunDataProvider.class);
    _objectStore = getService(ObjectStoreService.class);
    _objectStore.ensureStore(STORE);
    _objectStore.clearStore(STORE);
    _definitionPersistence = getService(DefinitionPersistence.class);
  }

  protected abstract String getWorkflow();

  /** initializes mapping. */
  protected abstract AnyMap initMapping();

  protected void checkBulks(final int expectedNumberOfBulks, final String bucket) throws Exception {
    final Collection<StoreObject> objects = _objectStore.getStoreObjectInfos(STORE, bucket);
    assertEquals(expectedNumberOfBulks, objects.size());
  }

  protected int getExpectedNumberOfBulks(final int numberOfRecords, final int maxRecordsPerBulk) {
    return numberOfRecords / maxRecordsPerBulk + numberOfRecords % maxRecordsPerBulk;
  }

  protected Any getIdColumns() {
    final AnySeq idColumns = DataFactory.DEFAULT.createAnySeq();
    idColumns.add("int_val");
    idColumns.add("float_val");
    return idColumns;
  }

  protected JobDefinition createJob(final String dbUrl, final String crawlSql, final String fetchSql,
    final long maxRecordsPerBulk) throws Exception {   
    return createJob(getName(), getWorkflow(), dbUrl, crawlSql, fetchSql, maxRecordsPerBulk);
  }

  protected JobDefinition createJob(final String jobName, final String workflow, final String dbUrl, final String crawlSql,
    final String fetchSql, final long maxRecordsPerBulk) throws Exception {
    final AnyMap job = DataFactory.DEFAULT.createAnyMap();
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("tempStore", STORE);
    parameters.put("store", STORE);
    parameters.put(ImportingConstants.TASK_PARAM_DATA_SOURCE, DATA_SOURCE);
    parameters.put(JdbcCrawlerWorker.TASK_PARAM_DB_URL, dbUrl);
    final AnyMap dbProps = DataFactory.DEFAULT.createAnyMap();
    dbProps.put("user", DB_USER);
    dbProps.put("password", DB_PASSWORD);
    parameters.put(JdbcCrawlerWorker.TASK_PARAM_DB_PROPS, dbProps);
    parameters.put(JdbcCrawlerWorker.TASK_PARAM_CRAWL_SQL, crawlSql);
    if (fetchSql != null) {
      parameters.put(JdbcFetcherWorker.TASK_PARAM_FETCH_SQL, fetchSql);
    }
    parameters.put(JdbcCrawlerWorker.TASK_PARAM_MAX_RECORDS_PER_BULK, maxRecordsPerBulk);
    parameters.put(JdbcCrawlerWorker.TASK_PARAM_ID_COLUMNS, getIdColumns());
    parameters.getSeq(JdbcCrawlerWorker.TASK_PARAM_DELTA_COLUMNS, true).add("bigint_val");
    parameters.put(ImportingConstants.TASK_PARAM_MAPPING, initMapping());
    job.put(JobDefinition.KEY_PARAMETERS, parameters);    
    job.put(JobDefinition.KEY_NAME, jobName);
    job.put(JobDefinition.KEY_WORKFLOW, workflow);
    return new JobDefinition(job);
  }

  protected void waitForJobRunCompleted(final String jobName, final String jobId, final long maxWaitTime)
    throws Exception {
    waitForJobRun(jobName, jobId, maxWaitTime, JobState.SUCCEEDED);
  }

  /** Waits for a job to be completed. */
  protected void waitForJobRun(final String jobName, final String jobId, final long maxWaitTime,
    final JobState expectedJobState) throws Exception {
    final long sleepTime = 500L;
    final long millisStarted = System.currentTimeMillis();
    while (true) {
      final AnyMap runData = _jobRunDataProvider.getJobRunData(jobName, jobId);
      final String jobRunState = runData.getStringValue(JobManagerConstants.DATA_JOB_STATE);
      if (jobRunState != null) {
        final JobState state = JobState.valueOf(jobRunState);
        if (state == expectedJobState) {
          return; // finally found what we're waiting for.
        }
        if (expectedJobState == JobState.SUCCEEDED) {
          assertNotSame(JobState.FAILED, state);
        }
      }
      assertTrue("Waited too long for job to complete", System.currentTimeMillis() - millisStarted <= maxWaitTime);
      Thread.sleep(sleepTime);
    }
  }

  protected void checkCrawledRecords(final int expectedNumberOfRecords, final String bucket,
    final Collection<String> expectedAttributes, final Collection<String> unexpectedAttributes,
    final Collection<String> expectedAttachments) throws ObjectStoreException, IOException {
    final Collection<StoreObject> objects = _objectStore.getStoreObjectInfos(STORE, bucket);
    int recordCount = 0;
    assertFalse(objects.isEmpty());
    for (final StoreObject objectInfo : objects) {
      final InputStream bulkStream = _objectStore.readObject(STORE, objectInfo.getId());
      try {
        final BinaryObjectStreamIterator records = new BinaryObjectStreamIterator(bulkStream);
        while (records.hasNext()) {
          final Record record = records.next();
          recordCount++;
          checkCrawledRecord(record, expectedAttributes, unexpectedAttributes, expectedAttachments);
        }
      } finally {
        IOUtils.closeQuietly(bulkStream);
      }
    }
    assertEquals(expectedNumberOfRecords, recordCount);
  }

  protected void checkCrawledRecord(final Record record, final Collection<String> expectedAttributes,
    final Collection<String> unexpectedAttributes, final Collection<String> expectedAttachments) {
    assertNotNull(record);
    assertNotNull(record.getId());
    assertEquals(DATA_SOURCE, record.getSource());
    final AnyMap metadata = record.getMetadata();
    if (expectedAttributes != null) {
      for (final String attrib : expectedAttributes) {
        assertTrue("expected attribute: " + attrib, metadata.containsKey(attrib));
      }
    }
    if (unexpectedAttributes != null) {
      for (final String attrib : unexpectedAttributes) {
        assertFalse("attribute not expected: " + attrib, metadata.containsKey(attrib));
      }
    }
    if (expectedAttachments == null) {
      assertFalse(record.hasAttachments());
    } else {
      for (final String attachment : expectedAttachments) {
        assertTrue("expected attachment: " + attachment, record.hasAttachment(attachment));
      }
    }
  }

}
