/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.jdbc.test;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.BinaryObjectStreamIterator;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.jdbc.JdbcCrawlerWorker;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.StoreObject;

public class TestJdbcCrawlerWorker extends JdbcCrawlerJobTestBase {

  private static final String BUCKET = "crawledRows";

  private static final String WORKFLOWNAME = "crawlDatabase";

  /** test crawling with a simple configuration. */
  public void testSimpleCrawling() throws Exception {
    final int maxRecordsPerBulk = 1000;
    final JobDefinition jobDefinition =
      createJob(getConnectUrl(), "SELECT * FROM " + DB_TABLE_NAME, null, maxRecordsPerBulk);
    _definitionPersistence.addJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), maxRecordsPerBulk), BUCKET);
    final Collection<String> expectedAttributes =
      Arrays.asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value", "my-double-value",
        "my-varchar-value");
    final Collection<String> unexpectedAttributes = Arrays.asList("float_val");
    final Collection<String> expectedAttachments = Arrays.asList("my-binary-attachment");
    checkCrawledRecords(getNumberOfRowsToCreate(), BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  /** test crawling with a simple configuration. */
  public void testCrawlingWithDeltaColumnNullValues() throws Exception {
    final int maxRecordsPerBulk = 1000;
    final JobDefinition jobDefinition =
      createJob(getConnectUrl(), "SELECT * FROM " + DB_TABLE_NAME, null, maxRecordsPerBulk);
    jobDefinition.getParameters().getSeq(JdbcCrawlerWorker.TASK_PARAM_DELTA_COLUMNS, true).add("null-column");
    _definitionPersistence.addJob(jobDefinition);
    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), maxRecordsPerBulk), BUCKET);
    final Collection<String> expectedAttributes =
      Arrays.asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value", "my-double-value",
        "my-varchar-value");
    final Collection<String> unexpectedAttributes = Arrays.asList("float_val");
    final Collection<String> expectedAttachments = Arrays.asList("my-binary-attachment");
    checkCrawledRecordsDeltaNull(getNumberOfRowsToCreate(), BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  /** test 'maxAttachmentSize' parameter. */
  public void testMaxAttachmentSize() throws Exception {
    final int maxRecordsPerBulk = 1000;
    final JobDefinition jobDefinition =
      createJob(getConnectUrl(), "SELECT * FROM " + DB_TABLE_NAME, null, maxRecordsPerBulk);
    // set max. attachment size to 1 byte -> attachment of the test will not be accepted 
    jobDefinition.getParameters().put(JdbcCrawlerWorker.TASK_PARAM_MAX_ATTACHMENT_SIZE, 1);
    _definitionPersistence.addJob(jobDefinition);

    final String jobRunId = _jobRunEngine.startJob(getName());
    waitForJobRunCompleted(getName(), jobRunId, DEFAULT_WAIT_TIME);

    checkBulks(getExpectedNumberOfBulks(getNumberOfRowsToCreate(), maxRecordsPerBulk), BUCKET);
    final Collection<String> expectedAttributes =
      Arrays.asList(ImportingConstants.ATTRIBUTE_DELTA_HASH, "my-int-value", "my-big-int-value", "my-double-value",
        "my-varchar-value");
    final Collection<String> unexpectedAttributes = Arrays.asList("float_val");
    final Collection<String> expectedAttachments = null; // -> test fails if records have attachments
    checkCrawledRecords(getNumberOfRowsToCreate(), BUCKET, expectedAttributes, unexpectedAttributes,
      expectedAttachments);
  }

  protected void checkCrawledRecordsDeltaNull(final int expectedNumberOfRecords, final String bucket,
    final Collection<String> expectedAttributes, final Collection<String> unexpectedAttributes,
    final Collection<String> expectedAttachments) throws ObjectStoreException, IOException {
    final Collection<StoreObject> objects = _objectStore.getStoreObjectInfos(STORE, bucket);
    int recordCount = 0;
    assertFalse(objects.isEmpty());
    for (final StoreObject objectInfo : objects) {
      final InputStream bulkStream = _objectStore.readObject(STORE, objectInfo.getId());
      try {
        final BinaryObjectStreamIterator records = new BinaryObjectStreamIterator(bulkStream);
        while (records.hasNext()) {
          final Record record = records.next();
          recordCount++;
          checkCrawledRecord(record, expectedAttributes, unexpectedAttributes, expectedAttachments);
          assertTrue(record.getMetadata().getStringValue(ImportingConstants.ATTRIBUTE_DELTA_HASH).endsWith("NULL"));
        }
      } finally {
        IOUtils.closeQuietly(bulkStream);
      }
    }
    assertEquals(expectedNumberOfRecords, recordCount);
  }

  @Override
  protected AnyMap initMapping() {
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    map.put("int_val", "my-int-value");
    map.put("bigint_val", "my-big-int-value");
    map.put("double_val", "my-double-value");
    // map.put("float_val", "my-file-extension"); // don't map to check if the value is excluded in the result record
    map.put("varchar_val", "my-varchar-value");
    map.put("varbinary_val", "my-binary-attachment");
    return map;
  }

  @Override
  protected String getWorkflow() {
    return WORKFLOWNAME;
  }

  /** @return the number of rows to create. */
  @Override
  protected int getNumberOfRowsToCreate() {
    return 10000;
  }

}
