/*******************************************************************************
 * Copyright (c) 2008, 2012 Empolis IM GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Empolis IM GmbH) - initial implementation
 *******************************************************************************/
package org.eclipse.smila.importing.crawler.web.test;

import junit.framework.TestCase;

import org.eclipse.smila.importing.crawler.web.utils.UriHelper;

/** Test for {@link UriHelper} class. */
public class TestUriHelper extends TestCase {

  /** tests makeAbsolute(). */
  public void testMakeAbsolute() throws Exception {
    assertEquals("http://www.test.de/path", UriHelper.makeAbsolute("http://www.test.de/", "path"));
    assertEquals("http://www.test.de/path", UriHelper.makeAbsolute("http://www.test.de", "/path"));
    assertEquals("http://www.test.de/path", UriHelper.makeAbsolute("http://www.test.de/index.html", "path"));
    assertEquals("http://www.test.de/path1", UriHelper.makeAbsolute("http://www.test.de/", "./path1"));
    assertEquals("http://www.test.de/path2", UriHelper.makeAbsolute("http://www.test.de/path1/", "../path2"));
    assertEquals("http://www.test.de/path?query", UriHelper.makeAbsolute("http://www.test.de/path", "?query"));
    assertEquals("http://www.test.de/path/index.html",
      UriHelper.makeAbsolute("http://www.test.de/path/", "index.html"));
    assertEquals("http://www.example.com/test",
      UriHelper.makeAbsolute("http://www.test.de/", "http://www.example.com/test"));
    assertNull(UriHelper.makeAbsolute("relative-base-link", "path"));
  }

  /** tests getHost(). */
  public void testGetHost() throws Exception {
    assertEquals("www.google.com", UriHelper.getHost("http://www.google.com"));
    assertEquals("www.google.com", UriHelper.getHost("http://www.google.com/bla"));
    assertEquals("www.google.com", UriHelper.getHost("https://www.google.com/bla/bla"));
    assertEquals("google.com", UriHelper.getHost("http://google.com/bla"));
    assertEquals("127.0.0.1", UriHelper.getHost("http://127.0.0.1/bla"));
  }

  /** tests getDomain(). */
  public void testGetDomain() throws Exception {
    assertEquals("google.com", UriHelper.getDomain("google.com"));
    assertEquals("google.com", UriHelper.getDomain("www.google.com"));
    assertEquals("my.google.com", UriHelper.getDomain("www.my.google.com"));
    assertEquals("google", UriHelper.getDomain("google"));
  }

  public void testGetHostAndPort() throws Exception {
    assertEquals("www.google.com:-1", UriHelper.getHostAndPort("http://www.google.com"));
    assertEquals("www.google.com:-1", UriHelper.getHostAndPort("http://www.google.com/bla"));
    assertEquals("www.google.com:-1", UriHelper.getHostAndPort("https://www.google.com/bla/bla"));
    assertEquals("www.google.com:81", UriHelper.getHostAndPort("https://www.google.com:81/bla/bla"));
    assertEquals("google.com:-1", UriHelper.getHostAndPort("http://google.com/bla"));
    assertEquals("google.com:80", UriHelper.getHostAndPort("http://google.com:80/bla"));
    assertEquals("127.0.0.1:-1", UriHelper.getHostAndPort("http://127.0.0.1/bla"));
    assertEquals("127.0.0.1:8080", UriHelper.getHostAndPort("http://127.0.0.1:8080/bla"));
  }

  public void testNormalize() throws Exception {
    assertEquals("http://www.google.com/bla/bla", UriHelper.normalizeUrl("", "http://www.GOOGLE.com/bla/bla"));
    assertEquals("https://www.google.com/bla/bla", UriHelper.normalizeUrl(null, "https://www.GOOGLE.com/bla/bla"));

    assertEquals("http://www.google.com/bla/bla", UriHelper.normalizeUrl("", "http://www.google.com:80/bla/bla"));
    assertEquals("https://www.google.com/bla/bla", UriHelper.normalizeUrl("", "https://www.google.com:443/bla/bla"));
    assertEquals("http://www.google.com:8080/bla/bla",
      UriHelper.normalizeUrl("", "http://www.google.com:8080/bla/bla"));
    assertEquals("https://www.google.com:80/bla/bla",
      UriHelper.normalizeUrl("", "https://www.google.com:80/bla/bla"));

    assertNull(UriHelper.normalizeUrl(null, "mailto:johndoe@example.com"));

    // we can deal with some unescaped links ...
    assertEquals("https://www.google.com:80/unescaped%20link.html",
      UriHelper.normalizeUrl(null, "https://www.google.com:80/unescaped link.html"));
    // we can deal with some unescaped links ...
    assertEquals("https://www.google.com:80/ünescäpedlönk.html",
      UriHelper.normalizeUrl("https://www.google.com:80/", "ünescäpedlönk.html"));
    // ... but not with all.
    assertNull(UriHelper.normalizeUrl(null, "relative link.html"));
    assertEquals("http://www.google.com/relative%20link.html",
      UriHelper.normalizeUrl("http://www.google.com/", "relative link.html"));

    // ignore invalid fragment part
    assertEquals("http://www.google.com/index.html",
      UriHelper.normalizeUrl("http://www.google.com/bla/", "../index.html#section a"));
    // these kind of links occur in javadocs: invalid because the backslash should be encoded.
    assertEquals("http://www.google.com/../../../../../index.html?package/%5Cclass-use.html",
      UriHelper.normalizeUrl("http://www.google.com", "../../../../../index.html?package/\\class-use.html"));
  }
}
