/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.web.test;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.http.server.HttpService;
import org.eclipse.smila.importing.crawler.web.test.util.RobotsTxtUtil;
import org.eclipse.smila.importing.util.PropertyNameMapper;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.taskmanager.TaskCounter;
import org.eclipse.smila.taskmanager.TaskManager;

public abstract class WebWorkerTestBase extends WebHelperTestBase {

  protected static final String STORENAME = "buckets";

  protected static final String TESTSERVER = "http://localhost:8765/";

  protected static final String BASEURL = TESTSERVER + "files/";

  protected JobRunEngine _jobRunEngine;

  protected JobRunDataProvider _jobRunDataProvider;

  protected DefinitionPersistence _defPersistence;

  protected ObjectStoreService _objectStore;

  protected BulkbuilderService _bulkbuilder;

  protected TaskManager _taskManager;

  protected final PropertyNameMapper _mapper = _webCrawlingContext.getMapper();

  public WebWorkerTestBase() {
    super();
  }

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _jobRunEngine = getService(JobRunEngine.class);
    _jobRunDataProvider = getService(JobRunDataProvider.class);
    _defPersistence = getService(DefinitionPersistence.class);
    _objectStore = getService(ObjectStoreService.class);
    resetStore();
    _bulkbuilder = getService(BulkbuilderService.class);
    _taskManager = getService(TaskManager.class);
    getService(HttpService.class); // wait for system started.
    RobotsTxtUtil.deleteRobotsTxt();
    RobotsTxtUtil.postRobotsTxtException("");
  }

  @Override
  protected void tearDown() throws Exception {
    RobotsTxtUtil.deleteRobotsTxt();
    RobotsTxtUtil.postRobotsTxtException("");
    super.tearDown();
  }

  /** ensure that store is empty. */
  protected void resetStore() throws Exception {
    _objectStore.removeStore(STORENAME);
    _objectStore.ensureStore(STORENAME);
  }

  /** waits until all tasks are finished. */
  protected void waitForTasksFinished(final int maxWaitTime) throws Exception {
    final long sleepTime = 500L;
    final long millisStarted = System.currentTimeMillis();
    while (true) {
      final Map<String, TaskCounter> counters = _taskManager.getTaskCounters();
      boolean haveTasks = false;
      for (final TaskCounter counter : counters.values()) {
        haveTasks |= counter.getTasksInProgress() > 0 || counter.getTasksTodo() > 0;
      }
      if (!haveTasks) {
        return;
      }
      assertTrue("Waited too long for job to complete", System.currentTimeMillis() - millisStarted <= maxWaitTime);
      Thread.sleep(sleepTime);
    }
  }

  /** get bulks from bucket sorted by IDs. assert that the number is as expected. */
  protected List<StoreObject> getSortedBulks(final String bucket, final int expectedNumberOfBulks)
    throws ObjectStoreException {
    final List<StoreObject> bulks = new ArrayList<StoreObject>(_objectStore.getStoreObjectInfos(STORENAME, bucket));
    assertEquals(expectedNumberOfBulks, bulks.size());
    Collections.sort(bulks, new Comparator<StoreObject>() {
      @Override
      public int compare(final StoreObject o1, final StoreObject o2) {
        return o1.getId().compareTo(o2.getId());
      }
    });
    return bulks;
  }

}
