/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.importing.crawler.web.test.httphandler;

import java.io.IOException;
import java.net.InetAddress;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;
import java.util.regex.Pattern;

import org.apache.commons.io.IOUtils;
import org.eclipse.jetty.http.HttpMethods;
import org.eclipse.jetty.http.HttpStatus;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.util.ARequestHandler;
import org.eclipse.smila.http.server.util.UnsupportedMethodException;
import org.eclipse.smila.importing.crawler.web.test.AllTests;
import org.eclipse.smila.utils.config.ConfigUtils;

/** A simple handler for testing HTTP redirects */
public class HostReplaceHandler extends ARequestHandler {

  public static final String I_DO_NOT_EXIST = "i-do-not-exist.";

  private static final String PLACE_HOLDER_SAME_HOST = "[XXX]";

  private static final String PLACE_HOLDER_SAME_DOMAIN = "[YXX]";

  @Override
  public void handle(final HttpExchange exchange) {
    boolean closeConnection = false;
    String resultHtml = null;
    String fileDate = null;
    final String method = exchange.getRequestMethod();
    final String requestUri = exchange.getRequestURI();
    try {
      try {
        if (!HttpMethods.GET.equals(method)) {
          throw new UnsupportedMethodException("Unsupported http method '" + method + "'");
        }
        final String fileName = requestUri.substring(requestUri.lastIndexOf('/') + 1);
        final String canonicalHostName = InetAddress.getLocalHost().getCanonicalHostName().toLowerCase();
        final String baseUrlThisHost =
          "http://" + canonicalHostName + ':' + exchange.getPort()
            + requestUri.substring(0, requestUri.lastIndexOf('/'));
        final String baseUrlOtherHostofSameDomain =
          "http://" + canonicalHostName.replaceFirst("[^\\.]*\\.", I_DO_NOT_EXIST) + ':' + exchange.getPort()
            + requestUri.substring(0, requestUri.lastIndexOf('/'));
        resultHtml = getFileContent(fileName);
        resultHtml = resultHtml.replaceAll(Pattern.quote(PLACE_HOLDER_SAME_HOST), baseUrlThisHost);
        resultHtml = resultHtml.replaceAll(Pattern.quote(PLACE_HOLDER_SAME_DOMAIN), baseUrlOtherHostofSameDomain);
        fileDate = getFileDate(fileName);

      } catch (final Throwable error) {
        _log.warn("Error processing " + method + " " + requestUri, error);
        closeConnection = prepareResponseHeader(exchange, 500, false, 0, null);
        return; // to skip writing of success result.
      }

      closeConnection = writeSuccessResult(exchange, resultHtml, resultHtml.getBytes("utf-8").length, fileDate);
    } catch (final IOException ex) {
      _log.warn("Error writing the response. Maybe client has closed the connection.", ex);
      closeConnection = true;
    } finally {
      try {
        if (closeConnection) {
          IOUtils.closeQuietly(exchange.getResponseStream());
        } else {
          exchange.getResponseStream().flush();
        }
      } catch (final Exception ex) {
        ; // if there is an exception it should have occurred before and be logged and everything.
      }
    }
  }

  public boolean prepareResponseHeader(final HttpExchange exchange, final int statusCode,
    final boolean hasResponseBody, final int contentLength, final String lastModified) {
    boolean closeConnection = false;
    exchange.setResponseStatus(statusCode);
    final String connectionHeader = exchange.getRequestHeader("Connection");
    if (connectionHeader != null && connectionHeader.equals("close")) {
      closeConnection = true;
      exchange.setResponseHeader("Connection", "close");
    } else {
      exchange.setResponseHeader("Connection", "keep-alive");
      exchange.setResponseHeader("Keep-Alive", "timeout=0");
    }
    if (hasResponseBody && statusCode != HttpStatus.NO_CONTENT_204) {
      exchange.setResponseHeader("Content-Type", "text/html; charset=utf-8");
      exchange.setResponseHeader("Content-Length", String.valueOf(contentLength));
      if (lastModified != null) {
        exchange.setResponseHeader("Last-Modified", lastModified);
      }
    } else {
      exchange.setResponseHeader("Content-Length", "0");
    }
    return closeConnection;
  }

  private final boolean writeSuccessResult(final HttpExchange exchange, final String resultHtml,
    final int contentLength, final String lastModified) throws IOException {
    final int statusCode = HttpStatus.OK_200;
    final boolean closeConnection =
      prepareResponseHeader(exchange, statusCode, resultHtml != null, contentLength, lastModified);
    if (resultHtml != null) {
      exchange.getResponseStream().write(resultHtml.getBytes());
    }
    return closeConnection;
  }

  /** get content of a test file from file immediately. */
  private String getFileContent(final String fileName) throws IOException {
    return IOUtils.toString(ConfigUtils.getConfigStream(AllTests.BUNDLE_ID, "files/" + fileName), "utf-8");
  }

  /** get las modified date of a test file from file immediately. */
  private String getFileDate(final String fileName) throws IOException {
    // Fri, 31 Aug 2012 10:46:18 GMT
    final DateFormat dateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zz", Locale.US);
    dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
    return dateFormat.format(new Date(ConfigUtils.getConfigFile(AllTests.BUNDLE_ID, "files/" + fileName)
      .lastModified()));
  }
}
