/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.web.test.httphandler;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;

import org.apache.commons.io.IOUtils;
import org.eclipse.jetty.http.HttpMethods;
import org.eclipse.jetty.http.HttpStatus;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.json.JsonHttpUtils;
import org.eclipse.smila.http.server.util.ARequestHandler;

/**
 * A simple test handler for storing and delivering robots.txt files.
 */
public class RobotsTxtHandler extends ARequestHandler {

  private String _robotsTxt;

  private Exception _exception;

  @Override
  public void handle(HttpExchange exchange) throws IOException {
    boolean closeConnection = false;
    final String method = exchange.getRequestMethod();
    final String requestUri = exchange.getRequestURI();
    try {
      try {
        if (HttpMethods.PUT.equals(method)) {
          final String in = readInput(exchange);
          _robotsTxt = in;
          closeConnection = writeSuccessResult(exchange, null);
        } else if (HttpMethods.POST.equals(method)) {
          // POST is used for provoking exceptions, exCode is e.g. "401"
          final String exCode = readInput(exchange);
          if (exCode == null || "".equals(exCode)) {
            _exception = null; // reset exception            
          } else {
            _exception = new RuntimeException(exCode);
          }
        } else if (HttpMethods.DELETE.equals(method)) {
          _robotsTxt = null;
          closeConnection = writeSuccessResult(exchange, null);
        } else { // GET
          if (_exception != null) {
            throw _exception;
          }
          if (_robotsTxt == null) {
            throw new FileNotFoundException();
          }
          closeConnection = writeSuccessResult(exchange, _robotsTxt);
        }

      } catch (final Throwable e) {
        _log.warn("Error processing " + method + " " + requestUri, e);
        writeErrorResult(exchange, e);
      }

    } catch (final IOException ex) {
      _log.warn("Error writing the response. Maybe client has closed the connection.", ex);
      closeConnection = true;
    } finally {
      try {
        if (closeConnection) {
          IOUtils.closeQuietly(exchange.getResponseStream());
        } else {
          exchange.getResponseStream().flush();
        }
      } catch (final Exception ex) {
        ; // if there is an exception it should have occurred before and be logged and everything.
      }
    }
  }

  /** read HTTP body. */
  public String readInput(final HttpExchange exchange) {
    InputStream in = null;
    try {
      in = exchange.getRequestStream();
      final StringWriter writer = new StringWriter();
      IOUtils.copy(in, writer, "utf-8");
      final String result = writer.toString();
      return result;
    } catch (final Exception e) {
      throw new RuntimeException("Error reading input", e);
    } finally {
      IOUtils.closeQuietly(in);
    }
  }

  /** read HTTP response header. */
  public boolean prepareResponseHeader(final HttpExchange exchange, final int statusCode,
    final boolean hasResponseBody, final int contentLength) {
    boolean closeConnection = false;
    exchange.setResponseStatus(statusCode);
    final String connectionHeader = exchange.getRequestHeader("Connection");
    if (connectionHeader != null && connectionHeader.equals("close")) {
      closeConnection = true;
      exchange.setResponseHeader("Connection", "close");
    } else {
      exchange.setResponseHeader("Connection", "keep-alive");
      exchange.setResponseHeader("Keep-Alive", "timeout=0");
    }
    if (hasResponseBody && statusCode != HttpStatus.NO_CONTENT_204) {
      exchange.setResponseHeader("Content-Type", "text/plain; charset=utf-8");
      exchange.setResponseHeader("Content-Length", String.valueOf(contentLength));
    } else {
      exchange.setResponseHeader("Content-Length", "0");
    }
    return closeConnection;
  }

  /** helper method for a HTTP result with the correct error status code. */
  private boolean writeErrorResult(final HttpExchange exchange, final Throwable e) throws IOException {
    int errorCode;
    if (e instanceof RuntimeException && "401".equals(e.getMessage())) {
      errorCode = 401;
    } else if (e instanceof RuntimeException && "403".equals(e.getMessage())) {
      errorCode = 403;
    } else if (e instanceof FileNotFoundException) {
      errorCode = 404;
    } else {
      errorCode = new JsonHttpUtils().getErrorStatus(e);
    }
    final boolean closeConnection = prepareResponseHeader(exchange, errorCode, false, -1);
    return closeConnection;
  }

  /** helper method for a HTTP result with status code 200. */
  private boolean writeSuccessResult(final HttpExchange exchange, final String resultString) throws IOException {
    final int statusCode = HttpStatus.OK_200;
    boolean closeConnection;
    if (resultString != null) {
      final byte[] content = resultString.getBytes("utf-8");
      closeConnection = prepareResponseHeader(exchange, statusCode, true, content.length);
      exchange.getResponseStream().write(content);
    } else {
      closeConnection = prepareResponseHeader(exchange, statusCode, false, -1);
    }
    return closeConnection;
  }

}
