/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.state.objectstore.test;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.RequestHandler;
import org.eclipse.smila.importing.DeltaService;
import org.eclipse.smila.test.DeclarativeServiceTestCase;
import org.eclipse.smila.utils.http.NotFoundHTTPResult;

/** some simple tests for the ReST admin handlers for DeltaService. */
public class TestHttpHandlers extends DeclarativeServiceTestCase {
  /** used to add data to test. */
  private DeltaService _service;

  /** delta admin handler to test. */
  private JsonRequestHandler _deltaAdmin;

  /** delta source admin handler to test. */
  private JsonRequestHandler _sourceAdmin;

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _service = getService(DeltaService.class);
    _service.clearAll();
    _deltaAdmin = (JsonRequestHandler) getService(RequestHandler.class, "(uriPattern=/importing/delta/?$)");
    _sourceAdmin =
      (JsonRequestHandler) getService(RequestHandler.class, "(uriPattern=/importing/delta/\\([^/]+\\)/?$)");
  }

  /** test GET with no current sources. */
  public void testAdminHandlerGetEmpty() throws Exception {
    final AnyMap response = (AnyMap) _deltaAdmin.process("GET", "/smila/importing/delta", null, null);
    assertNotNull(response);
    assertTrue(response.containsKey("sources"));
    assertTrue(response.getSeq("sources").isEmpty());
  }

  /** test GET with one current source. */
  public void testAdminHandlerGetNotEmpty() throws Exception {
    final String recordId = "testAdminHandlerGetNotEmpty";
    _service.markAsUpdated("s1", recordId, "j1", "h1");
    final AnyMap response = (AnyMap) _deltaAdmin.process("GET", "/smila/importing/delta", null, null);
    final AnySeq sources = response.getSeq("sources");
    assertEquals(1, sources.size());
    assertEquals("s1", sources.getMap(0).getStringValue("id"));
    assertTrue(sources.getMap(0).containsKey("url"));
  }

  /** test DELETE with two sources. */
  public void testAdminHandlerDelete() throws Exception {
    final String recordId = "testAdminHandlerDelete";
    _service.markAsUpdated("s1", recordId, "j1", "h1");
    _service.markAsUpdated("s2", recordId, "j1", "h1");
    assertNull(_deltaAdmin.process("DELETE", "/smila/importing/delta", null, null));
    assertTrue(_service.getSourceIds().isEmpty());
  }

  /** test GET with an unknown source name. */
  public void testSourceHandlerGetUnknown() throws Exception {
    try {
      _sourceAdmin.process("GET", "/smila/importing/delta/unknown/", null, null);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue(ex instanceof NotFoundHTTPResult || ex.getCause() instanceof NotFoundHTTPResult);
    }
  }

  /** test GET with an existing source name. */
  public void testSourceHandlerGetExisting() throws Exception {
    final String recordId = "testSourceHandlerGetExisting";
    _service.markAsUpdated("s1", recordId, "j1", "h1");
    final AnyMap response = (AnyMap) _sourceAdmin.process("GET", "/smila/importing/delta/s1", null, null);
    assertNotNull(response);
    assertEquals("s1", response.getStringValue("id"));
    assertTrue(response.containsKey("count"));
  }

  /** test DELETE with an existing source name. */
  public void testSourceHandlerDelete() throws Exception {
    final String recordId = "testSourceHandlerDelete";
    _service.markAsUpdated("s1", recordId, "j1", "h1");
    _service.markAsUpdated("s2", recordId, "j1", "h1");
    assertNull(_sourceAdmin.process("DELETE", "/smila/importing/delta/s1", null, null));
    assertEquals(1, _service.getSourceIds().size());
    assertEquals("s2", _service.getSourceIds().iterator().next());
  }
}
