/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.worker;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.DeltaException;
import org.eclipse.smila.importing.DeltaImportStrategy;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.State;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.input.Inputs;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.output.Outputs;
import org.eclipse.smila.taskworker.output.RecordOutput;

/**
 * Worker implementation that performs delta checks.
 * 
 * @author stuc07
 * 
 */
public class DeltaCheckerWorker extends WorkerUsingDeltaService {

  private static final String WORKER_NAME = "deltaChecker";

  private static final String INPUT_SLOT_NAME = "recordsToCheck";

  private static final String OUTPUT_SLOT_RECORDS = "updatedRecords";

  private static final String OUTPUT_SLOT_COMPOUNDS = "updatedCompounds";

  private final Log _log = LogFactory.getLog(getClass());

  @Override
  public String getName() {
    return WORKER_NAME;
  }

  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    final String jobRunId = getJobRunId(taskContext);
    final Inputs inputs = taskContext.getInputs();
    final RecordInput recordInput = inputs.getAsRecordInput(INPUT_SLOT_NAME);
    final Outputs outputs = taskContext.getOutputs();
    final RecordOutput recordOutput = outputs.getAsRecordOutput(OUTPUT_SLOT_RECORDS);
    final boolean hasCompoundOutput = outputs.getDataObjectCount(OUTPUT_SLOT_COMPOUNDS) > 0;
    int compoundBulkCount = 0;
    final DeltaImportStrategy deltaStrategy = getDeltaImportStrategy(taskContext);
    Record record = recordInput.getRecord();
    while (record != null && !taskContext.isCanceled()) {
      final Record updatedRecord = checkRecord(taskContext, jobRunId, record, deltaStrategy);
      if (updatedRecord != null) {
        if (hasCompoundOutput && isCompound(updatedRecord)) {
          final RecordOutput compoundOutput = outputs.getAsRecordOutput(OUTPUT_SLOT_COMPOUNDS, compoundBulkCount);
          compoundOutput.writeRecord(updatedRecord);
          compoundOutput.commit();
          compoundBulkCount++;
        } else {
          recordOutput.writeRecord(updatedRecord);
        }
      }
      record = recordInput.getRecord();
    }
  }

  /** get delta hash value from record and check with DeltaService. */
  private Record checkRecord(final TaskContext taskContext, final String jobRunId, final Record record,
    final DeltaImportStrategy deltaStrategy) throws DeltaException, ObjectStoreException, IOException {
    if (deltaStrategy.isDeltaCheckDisabled()) {
      if (_log.isDebugEnabled()) {
        _log.debug("Delta checking is disabled, record " + record.getId() + " is added to output.");
      }
      return record;
    }

    final String deltaHash = record.getMetadata().getStringValue(ImportingConstants.ATTRIBUTE_DELTA_HASH);
    if (deltaHash == null) {
      if (_log.isDebugEnabled()) {
        _log.debug("Record " + record.getId() + " has no hash attribute value, is added to output.");
      }
      return record;
    }

    final State deltaStatus = checkDeltaStateTimed(jobRunId, record, deltaHash, taskContext);
    if (deltaStatus == State.UPTODATE) {
      if (_log.isDebugEnabled()) {
        _log.debug("Record " + record.getId() + " is up to date");
      }
      if (isCompound(record)) {
        markCompoundElementsVisitedTimed(jobRunId, record, taskContext);
      }
      return null;
    }

    if (_log.isDebugEnabled()) {
      _log.debug("Record " + record.getId() + " will be added to output, has status " + deltaStatus);
    }
    return record;
  }

}
