/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.ipc.test;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Arrays;

import junit.framework.TestCase;

import org.eclipse.smila.ipc.IpcToken;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.ipc.bon.BinaryStreamReader;
import org.eclipse.smila.utils.config.ConfigUtils;

/**
 * Test BON reader.
 * 
 * @author scum36
 * 
 */
public class TestBinaryStreamReader extends TestCase {
  private final BinaryFactory _factory = new BinaryFactory();

  public void testReadStringValue() throws Exception {
    final BinaryStreamReader reader = createBonReader("15 08 c3 a4 68 6e 6c 69 63 68");
    assertSame(IpcToken.SCALAR_STRING, reader.nextToken());
    assertEquals("ähnlich", reader.currentStringValue());
    assertNull(reader.nextToken());
  }

  public void testReadLongValue() throws Exception {
    final BinaryStreamReader reader = createBonReader("09 8e 0c");
    assertSame(IpcToken.SCALAR_INT, reader.nextToken());
    assertEquals(-36364, reader.currentLongValue());
    assertNull(reader.nextToken());
  }

  public void testReadBooleanValue() throws Exception {
    final BinaryStreamReader reader = createBonReader("10");
    assertSame(IpcToken.SCALAR_BOOL, reader.nextToken());
    assertTrue(reader.currentBoolValue());
    assertNull(reader.nextToken());
  }

  public void testReadSequence() throws Exception {
    final BinaryStreamReader reader = createBonReader("1d 15 01 30 1e");
    assertSame(IpcToken.SEQUENCE_START, reader.nextToken());
    assertSame(IpcToken.SCALAR_STRING, reader.nextToken());
    assertEquals("0", reader.currentStringValue());
    assertSame(IpcToken.SEQUENCE_END, reader.nextToken());
    assertNull(reader.nextToken());
  }

  public void testReadMap() throws Exception {
    final BinaryStreamReader reader = createBonReader("1f 15 01 30 15 02 31 32 20");
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertSame(IpcToken.SCALAR_STRING, reader.nextToken());
    assertEquals("0", reader.currentStringValue());
    assertSame(IpcToken.SCALAR_STRING, reader.nextToken());
    assertEquals("12", reader.currentStringValue());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertNull(reader.nextToken());
  }

  public void testReadAttachment() throws Exception {
    final BinaryStreamReader reader = createBonReader("21 23 04 00 01 02 03 22");
    assertSame(IpcToken.ATTACHMENTS_START, reader.nextToken());
    assertSame(IpcToken.BINARY, reader.nextToken());
    assertTrue(Arrays.equals(new byte[] { 0, 1, 2, 3 }, reader.currentBinaryValue()));
    assertSame(IpcToken.ATTACHMENTS_END, reader.nextToken());
    assertNull(reader.nextToken());
  }

  public void testReadCustomTypeMap() throws Exception {
    final BinaryStreamReader reader = createBonReader("1f 2b 15 04 54 45 58 54 15 02 41 42 15 02 43 44 20");
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertSame(IpcToken.SCALAR_STRING, reader.nextToken());
    assertEquals("AB", reader.currentStringValue());
    assertSame(IpcToken.SCALAR_STRING, reader.nextToken());
    assertEquals("CD", reader.currentStringValue());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
  }

  /**
   * Parse a BON object with this content:
   * 
   * <pre>
   *   {
   *     "a": false,
   *     "string": "this is a string",
   *     "bon": [
   *         [
   *             {
   *                 "l": "literal1"
   *             }
   *         ],
   *         "delimiter",
   *         [
   *             {
   *                 "l": "literal2"
   *             },
   *             {
   *                 "x": {
   *                     "p": "pos1",
   *                     "s": "stem1",
   *                     "c": [
   *                         "component1"
   *                     ]
   *                 }
   *             }
   *         ],
   *         [
   *             {
   *                 "l": "literal3"
   *             },
   *             {
   *                 "x": {
   *                     "p": "pos2",
   *                     "s": "stem2",
   *                     "c": [
   *                         "component2",
   *                         "component3"
   *                     ]
   *                 }
   *             },
   *             {
   *                 "x": {
   *                     "p": "pos3",
   *                     "s": "stem3"
   *                 }
   *             }
   *         ]
   *     ],
   *     "number": 1,
   *     "z": true
   * }
   * </pre>
   */
  public void testReadObject() throws Exception {
    final InputStream is = ConfigUtils.getConfigStream(AllTests.BUNDLE_ID, "customtype.bon");
    final BinaryStreamReader reader = _factory.newStreamReader(is);
    assertSame(IpcToken.OBJECT_START, reader.nextToken());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());

    assertEquals("a", reader.nextStringValue());
    assertFalse(reader.nextBoolValue());

    assertEquals("string", reader.nextStringValue());
    assertEquals("this is a string", reader.nextStringValue());

    assertEquals("bon", reader.nextStringValue());
    assertSame(IpcToken.SEQUENCE_START, reader.nextToken());

    assertSame(IpcToken.SEQUENCE_START, reader.nextToken());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertEquals("l", reader.nextStringValue());
    assertEquals("literal1", reader.nextStringValue());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.SEQUENCE_END, reader.nextToken());

    assertEquals("delimiter", reader.nextStringValue());

    assertSame(IpcToken.SEQUENCE_START, reader.nextToken());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertEquals("l", reader.nextStringValue());
    assertEquals("literal2", reader.nextStringValue());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertEquals("x", reader.nextStringValue());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertEquals("p", reader.nextStringValue());
    assertEquals("pos1", reader.nextStringValue());
    assertEquals("s", reader.nextStringValue());
    assertEquals("stem1", reader.nextStringValue());
    assertEquals("c", reader.nextStringValue());
    assertSame(IpcToken.SEQUENCE_START, reader.nextToken());
    assertEquals("component1", reader.nextStringValue());
    assertSame(IpcToken.SEQUENCE_END, reader.nextToken());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.SEQUENCE_END, reader.nextToken());

    assertSame(IpcToken.SEQUENCE_START, reader.nextToken());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertEquals("l", reader.nextStringValue());
    assertEquals("literal3", reader.nextStringValue());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertEquals("x", reader.nextStringValue());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertEquals("p", reader.nextStringValue());
    assertEquals("pos2", reader.nextStringValue());
    assertEquals("s", reader.nextStringValue());
    assertEquals("stem2", reader.nextStringValue());
    assertEquals("c", reader.nextStringValue());
    assertSame(IpcToken.SEQUENCE_START, reader.nextToken());
    assertEquals("component2", reader.nextStringValue());
    assertEquals("component3", reader.nextStringValue());
    assertSame(IpcToken.SEQUENCE_END, reader.nextToken());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertEquals("x", reader.nextStringValue());
    assertSame(IpcToken.MAPPING_START, reader.nextToken());
    assertEquals("p", reader.nextStringValue());
    assertEquals("pos3", reader.nextStringValue());
    assertEquals("s", reader.nextStringValue());
    assertEquals("stem3", reader.nextStringValue());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.SEQUENCE_END, reader.nextToken());

    assertSame(IpcToken.SEQUENCE_END, reader.nextToken());

    assertEquals("number", reader.nextStringValue());
    assertEquals(1.0, reader.nextDoubleValue());

    assertEquals("z", reader.nextStringValue());
    assertTrue(reader.nextBoolValue());

    assertSame(IpcToken.MAPPING_END, reader.nextToken());
    assertSame(IpcToken.OBJECT_END, reader.nextToken());

    assertNull(reader.nextToken());
  }

  private BinaryStreamReader createBonReader(final String bonAsHexString) throws Exception {
    final String[] hexStrings = bonAsHexString.split(" ");
    final byte[] bonData = new byte[hexStrings.length];
    for (int i = 0; i < hexStrings.length; i++) {
      bonData[i] = (byte) Integer.parseInt(hexStrings[i], 16);
    }
    return _factory.newStreamReader(new ByteArrayInputStream(bonData));
  }
}
