/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH.  
 * All rights reserved. This program and the accompanying materials are made available
 * under the terms of the Eclipse Public License v1.0 which accompanies this distribution, 
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation               
 **********************************************************************************************************************/

package org.eclipse.smila.ipc.test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Arrays;

import junit.framework.TestCase;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.ipc.IpcStreamReader;
import org.eclipse.smila.ipc.IpcStreamWriter;
import org.eclipse.smila.ipc.IpcToken;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.ipc.json.JsonFactory;

/**
 * @author aweber
 */
public class TestIpcStreamWriter extends TestCase {

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** the factory to create JSON reader/writer objects. */
  private final JsonFactory _jsonFactory = new JsonFactory();

  /** the factory to create BON reader/writer objects. */
  private final BinaryFactory _binaryFactory = new BinaryFactory();

  /** the expected tokens in the test. */
  private final IpcToken[] _expectedTokens = new IpcToken[] { IpcToken.OBJECT_START, IpcToken.MAPPING_START,
    IpcToken.SCALAR_STRING, IpcToken.MAPPING_START, IpcToken.SCALAR_STRING, IpcToken.MAPPING_START,
    IpcToken.SCALAR_STRING, IpcToken.SCALAR_DOUBLE, IpcToken.MAPPING_END, IpcToken.SCALAR_STRING,
    IpcToken.SEQUENCE_START, IpcToken.SCALAR_STRING, IpcToken.SCALAR_STRING, IpcToken.SCALAR_INT,
    IpcToken.SCALAR_INT, IpcToken.SCALAR_STRING, IpcToken.SCALAR_STRING, IpcToken.SCALAR_INT, IpcToken.SCALAR_INT,
    IpcToken.SEQUENCE_END, IpcToken.MAPPING_END, IpcToken.MAPPING_END, IpcToken.OBJECT_END };

  /** the expected tokens in the test containg three attachments. */
  private final IpcToken[] _expectedTokensWithAttachments = new IpcToken[] { IpcToken.OBJECT_START,
    IpcToken.MAPPING_START, IpcToken.SCALAR_STRING, IpcToken.MAPPING_START, IpcToken.SCALAR_STRING,
    IpcToken.MAPPING_START, IpcToken.SCALAR_STRING, IpcToken.SCALAR_DOUBLE, IpcToken.MAPPING_END,
    IpcToken.SCALAR_STRING, IpcToken.SEQUENCE_START, IpcToken.SCALAR_STRING, IpcToken.SCALAR_STRING,
    IpcToken.SCALAR_INT, IpcToken.SCALAR_INT, IpcToken.SCALAR_STRING, IpcToken.SCALAR_STRING, IpcToken.SCALAR_INT,
    IpcToken.SCALAR_INT, IpcToken.SEQUENCE_END, IpcToken.MAPPING_END, IpcToken.MAPPING_END,
    IpcToken.ATTACHMENTS_START, IpcToken.SCALAR_STRING, IpcToken.BINARY, IpcToken.SCALAR_STRING, IpcToken.BINARY,
    IpcToken.SCALAR_STRING, IpcToken.BINARY, IpcToken.ATTACHMENTS_END, IpcToken.OBJECT_END };

  /** the expected BON values in the test. */
  private final String[] _expectedBinaryValues = new String[] { null, null, "result", null, "state", null, "code",
    "3.14", null, "annotations", null, "Sentence", "", "0", "13", "Token", "Heute", "3", "123456", null, null,
    null, null };

  /** the expected BON values in the test containg three attachments. */
  private final String[] _expectedBinaryValuesWithAttachments = new String[] { null, null, "result", null, "state",
    null, "code", "3.14", null, "annotations", null, "Sentence", "", "0", "13", "Token", "Heute", "3", "123456",
    null, null, null, null, "Attachment-1", new String(new byte[] { 1, 2 }), "Attachment-2",
    new String(new byte[] { 3, 4, 5 }), "Attachment-3", new String(new byte[] { 6, 7, 8, 9 }), null, null };

  /** the expected JSON values in the test. */
  private final String[] _expectedJsonValues = new String[] { null, "{", "result", "{", "state", "{", "code",
    "3.14", "}", "annotations", "[", "Sentence", "", "0", "13", "Token", "Heute", "3", "123456", "]", "}", "}",
    null };

  /**
   * test BinaryStreamReader/-Writer.
   */
  public void testBinary() throws IOException {
    doTestBinary(false);
  }

  /**
   * test BinaryStreamReader/-Writer with attachments.
   */
  public void testBinaryWithAttachments() throws IOException {
    doTestBinary(true);
  }

  /** main binary testing method. */
  private void doTestBinary(final boolean withAttachments) throws IOException {
    final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
    // write JSON output
    final IpcStreamWriter writer = _binaryFactory.newStreamWriter(outStream);
    writeOutputStart(writer);
    writeOutputMetadata(writer);
    if (withAttachments) {
      writeOutputAttachments(writer);
    }
    writeOutputEnd(writer);
    writer.close();
    // print DEBUG
    final byte[] content = outStream.toByteArray();
    _log.info(Arrays.toString(content));
    // read JSON input
    final ByteArrayInputStream inStream = new ByteArrayInputStream(outStream.toByteArray());
    if (withAttachments) {
      readInputBinary(_binaryFactory.newStreamReader(inStream), _expectedTokensWithAttachments,
        _expectedBinaryValuesWithAttachments);
    } else {
      readInputBinary(_binaryFactory.newStreamReader(inStream), _expectedTokens, _expectedBinaryValues);
    }
  }

  /** helper method to read some binary input. */
  private void readInputBinary(final IpcStreamReader reader, final IpcToken[] expectedTokens,
    final String[] expectedValues) throws IOException {
    IpcToken token = null;
    int i = 0;
    do {
      token = reader.nextToken();
      _log.info(token + " - " + reader.currentStringValue());
      assertEquals(expectedTokens[i], token);
      assertEquals(expectedValues[i], reader.currentStringValue());
      i++;
    } while (token != IpcToken.OBJECT_END);
  }

  /**
   * test with JsonStreamReader/-Writer.
   */
  public void testJson() throws IOException {
    doTestJson(false);
  }

  /**
   * test JsonStreamReader/-Writer with attachments -> attachments will be ignored.
   */
  public void testJsonWithAttachments() throws IOException {
    doTestJson(true);
  }

  /** main json testing method. */
  private void doTestJson(final boolean withAttachments) throws IOException {
    final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
    // write JSON output
    final IpcStreamWriter writer = _jsonFactory.newStreamWriter(outStream);
    writeOutputStart(writer);
    writeOutputMetadata(writer);
    if (withAttachments) {
      writeOutputAttachments(writer);
    }
    writeOutputEnd(writer);
    writer.close();
    // print DEBUG
    _log.info(outStream.toString("utf-8"));
    // read JSON input
    final ByteArrayInputStream inStream = new ByteArrayInputStream(outStream.toByteArray());
    readInputJson(_jsonFactory.newStreamReader(inStream));
  }

  /** helper method to read some json input. */
  private void readInputJson(final IpcStreamReader reader) throws IOException {
    IpcToken token = null;
    int i = 0;
    do {
      token = reader.nextToken();
      final String stringValue = reader.currentStringValue();
      _log.info(token + " - " + stringValue);
      assertEquals(_expectedTokens[i], token);
      assertEquals(_expectedJsonValues[i], stringValue);
      i++;
    } while (token != IpcToken.OBJECT_END);
  }

  /** helper method to write output start. */
  private void writeOutputStart(final IpcStreamWriter writer) throws IOException {
    writer.writeObjectStart(); // should have no effect for JSON
  }

  /** helper method to write output end. */
  private void writeOutputEnd(final IpcStreamWriter writer) throws IOException {
    writer.writeObjectEnd(); // should have no effect for JSON
  }

  /** helper method to write output metadata. */
  private void writeOutputMetadata(final IpcStreamWriter writer) throws IOException {
    writer.writeMappingStart();
    writer.writeMappingKey("result");
    writer.writeMappingStart();
    writer.writeMappingKey("state");
    writer.writeMappingStart();
    writer.writeMappingKey("code");
    writer.writeScalarDouble(3.14);
    writer.writeMappingEnd();
    writer.writeMappingKey("annotations");
    writer.writeSequenceStart();
    writer.writeScalarString("Sentence");
    writer.writeScalarString("");
    writer.writeScalarInt(0);
    writer.writeScalarInt(13);
    writer.writeScalarString("Token");
    writer.writeScalarString("Heute");
    writer.writeScalarInt(3);
    writer.writeScalarInt(123456);
    writer.writeSequenceEnd();
    writer.writeMappingEnd();
    writer.writeMappingEnd();
  }

  /** helper method to write output attachments. */
  private void writeOutputAttachments(final IpcStreamWriter writer) throws IOException {
    writer.writeAttachmentsStart();
    writer.writeScalarString("Attachment-1");
    writer.writeBinary(new byte[] { 1, 2 });
    writer.writeScalarString("Attachment-2");
    writer.writeBinary(new byte[] { 3, 4, 5 });
    writer.writeScalarString("Attachment-3");
    writer.writeBinary(new byte[] { 6, 7, 8, 9 });
    writer.writeAttachmentsEnd();
  }

}
