/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.jdbc.test;

import java.sql.Connection;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.jdbc.JdbcLoggingPipelet;
import org.eclipse.smila.jdbc.JdbcProvider;
import org.eclipse.smila.jdbc.JdbcWriterService;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.PipeletTracker;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/** Test for {@link TestJdbcLoggingPipelet} class. */
public class TestJdbcLoggingPipelet extends DeclarativeServiceTestCase {

  private static final String DB_URL = "jdbc:derby:memory:testpipelet";

  private static final List<Value> NO_VALUES = Collections.emptyList();

  private static final int WAIT_SECONDS = 2;

  private Blackboard _bb;

  private JdbcProvider _provider;

  private final AnyMap _dbProps = DataFactory.DEFAULT.createAnyMap();

  private Connection _conn;

  private JdbcLoggingPipelet _pipelet;

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _bb = getService(BlackboardFactory.class).createTransientBlackboard();

    _provider = getService(JdbcProvider.class);
    _dbProps.put(JdbcWriterService.DB_PROPERTY_USER_NAME, "user");
    _dbProps.put(JdbcWriterService.DB_PROPERTY_USER_PASSWORD, "password");

    _conn = TestJdbcWriterService.prepareConnection(_provider, DB_URL);

    final PipeletTracker tracker = getService(PipeletTracker.class);
    final Class<? extends Pipelet> pipeletClass =
      tracker.getRegisteredPipelets().get(JdbcLoggingPipelet.class.getName());
    _pipelet = (JdbcLoggingPipelet) pipeletClass.newInstance();

    final AnyMap config = DataFactory.DEFAULT.createAnyMap();
    config.put(JdbcLoggingPipelet.PARAM_DB_URL, DB_URL);
    config.put(JdbcLoggingPipelet.PARAM_DB_PROPS, _dbProps);
    _pipelet.configure(config);
  }

  @Override
  protected void tearDown() throws Exception {
    _conn.close();
    super.tearDown();
  }

  public void testErrorMissingUrl() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    final Pipelet p = new JdbcLoggingPipelet();
    try {
      parameters.put(JdbcLoggingPipelet.PARAM_DB_PROPS, _dbProps);
      parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
      parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_String1");
      p.process(_bb, new String[] { "1" });
      fail("should not work");
    } catch (final ProcessingException ex) {
      assertTrue(ex.getMessage().contains(JdbcLoggingPipelet.PARAM_DB_URL));
    }
  }

  public void testErrorMissingProperties() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    final Pipelet p = new JdbcLoggingPipelet();
    try {
      parameters.put(JdbcLoggingPipelet.PARAM_DB_URL, DB_URL);
      parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
      parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_String1");
      p.process(_bb, new String[] { "1" });
      fail("should not work");
    } catch (final ProcessingException ex) {
      assertTrue(ex.getMessage().contains(JdbcLoggingPipelet.PARAM_DB_PROPS));
    }
  }

  public void testErrorInvalidProperties() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    final Pipelet p = new JdbcLoggingPipelet();
    try {
      parameters.put(JdbcLoggingPipelet.PARAM_DB_URL, DB_URL);
      parameters.put(JdbcLoggingPipelet.PARAM_DB_PROPS, "user=password");
      parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
      parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_String1");
      p.process(_bb, new String[] { "1" });
      fail("should not work");
    } catch (final ProcessingException ex) {
      assertTrue(ex.getMessage().contains(JdbcLoggingPipelet.PARAM_DB_PROPS));
    }
  }

  public void testErrorMissingStatement() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    final Pipelet p = new JdbcLoggingPipelet();
    try {
      parameters.put(JdbcLoggingPipelet.PARAM_DB_URL, DB_URL);
      parameters.put(JdbcLoggingPipelet.PARAM_DB_PROPS, _dbProps);
      parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_String1");
      p.process(_bb, new String[] { "1" });
      fail("should not work");
    } catch (final ProcessingException ex) {
      ; // ignore
    }
  }

  public void testErrorInvadidValuePaths() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    final Pipelet p = new JdbcLoggingPipelet();
    try {
      parameters.put(JdbcLoggingPipelet.PARAM_DB_URL, DB_URL);
      parameters.put(JdbcLoggingPipelet.PARAM_DB_PROPS, _dbProps);
      parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
      parameters.put(JdbcLoggingPipelet.PARAM_VALUE_PATHS, "Att_String1");
      p.process(_bb, new String[] { "1" });
      fail("should not work");
    } catch (final ProcessingException ex) {
      assertTrue(ex.getMessage().contains(JdbcLoggingPipelet.PARAM_VALUE_PATHS));
    }
  }

  public void testErrorInvalidValuePathType() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    final Pipelet p = new JdbcLoggingPipelet();
    try {
      parameters.put(JdbcLoggingPipelet.PARAM_DB_URL, DB_URL);
      parameters.put(JdbcLoggingPipelet.PARAM_DB_PROPS, _dbProps);
      parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
      parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add(1);
      p.process(_bb, new String[] { "1" });
      fail("should not work");
    } catch (final ProcessingException ex) {
      assertTrue(ex.getMessage().contains(JdbcLoggingPipelet.PARAM_VALUE_PATHS));
    }
  }

  public void testStringAttributes() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_String1");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_String2");
    record.put("Att_String1", "hello");
    record.put("Att_String2", "pipelet");
    _pipelet.process(_bb, new String[] { "1" });
    TestJdbcWriterService.checkStringEntry(_conn, "hello", "pipelet");
  }

  public void testUnsetAttributes() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_String1");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_String2");
    record.put("Att_String1", "hello");
    _pipelet.process(_bb, new String[] { "1" });
    TestJdbcWriterService.checkStringEntry(_conn, "hello", null);
  }

  public void testIntegerAttributes() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (int1) values (?)");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_Int1");
    for (int i = 0; i < 10; i++) {
      record.put("Att_Int1", i);
      _pipelet.process(_bb, new String[] { "1" });
    }
    TestJdbcWriterService.checkIntegerEntries(_conn, 10);
  }

  public void testValuePaths() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Sub/Att_String1");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Sub/Att_String2");
    record.getMap("Sub", true).put("Att_String1", "hello");
    record.getMap("Sub", true).put("Att_String2", "pipelet");
    _pipelet.process(_bb, new String[] { "1" });
    TestJdbcWriterService.checkStringEntry(_conn, "hello", "pipelet");
  }

  public void testMissingSubMap() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Sub1/Att_String1");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Sub2/Att_String2");
    record.getMap("Sub1", true).put("Att_String1", "hello");
    _pipelet.process(_bb, new String[] { "1" });
    TestJdbcWriterService.checkStringEntry(_conn, "hello", null);
  }

  public void testResolutionOfSequences() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Sub1/Att_String1");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Sub2/Att_String2");
    final AnyMap sub1 = DataFactory.DEFAULT.createAnyMap();
    sub1.put("Att_String1", "hello");
    record.add("Sub1", sub1);
    record.getMap("Sub2", true).getSeq("Att_String2", true).add("pipelet");
    _pipelet.process(_bb, new String[] { "1" });
    TestJdbcWriterService.checkStringEntry(_conn, "hello", "pipelet");
  }

  public void testEmptySequences() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (string1, string2) values (?, ?)");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Sub1/Att_String1");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Sub2/Att_String2");
    record.getMap("Sub1", true).put("Att_String1", "hello");
    record.getMap("Sub2", true).getSeq("Att_String2", true);
    _pipelet.process(_bb, new String[] { "1" });
    TestJdbcWriterService.checkStringEntry(_conn, "hello", null);
  }

  public void testErrorHandling() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    parameters.put(JdbcLoggingPipelet.PARAM_STMT, "invalid sql statement");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_String1");
    record.put("Att_String1", "hello");
    final String[] recordIds = new String[] { "1" };
    final String[] resultIds = _pipelet.process(_bb, recordIds);
    assertTrue(Arrays.equals(recordIds, resultIds));
  }

  public void testQueueFullErrorHandling() throws Exception {
    final AnyMap record = _bb.getRecord("1", Get.NEW).getMetadata();
    final AnyMap parameters = record.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true);
    parameters.put(JdbcLoggingPipelet.PARAM_STMT, "insert into entries (int1) values (?)");
    parameters.getSeq(JdbcLoggingPipelet.PARAM_VALUE_PATHS, true).add("Att_Int1");
    int numberOfEntries = 0;
    try {
      for (numberOfEntries = 0; numberOfEntries < 10000; numberOfEntries++) {
        record.put("Att_Int1", numberOfEntries);
        _pipelet.process(_bb, new String[] { "1" });
      }
      fail("should not work");
    } catch (final ProcessingException ex) {
      ; //
    }
    assertTrue(numberOfEntries >= 100);
    TestJdbcWriterService.checkIntegerEntries(_conn, numberOfEntries);
  }
}
