/*******************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (Empolis Information Management GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.jdbc;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.common.logging.MessageCollector;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;
import org.eclipse.smila.utils.service.ServiceUtils;

/**
 * Pipelet that fetches data from a database via JDBC, using a statement which is parameterized by record values.
 * 
 * The values for the PreparedStatement are selected from the input record by following the configured value paths.
 * 
 * If a value path references a single value, this is used. If it references a sequence of values, the first value is
 * used. In any other case, the value is set 'null'.
 */
public class JdbcFetcherPipelet extends AbstractJdbcPipelet {

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** OSGI service used to log to database. */
  private JdbcAccessService _jdbcAccess;

  @Override
  public String[] process(final Blackboard blackboard, final String[] recordIds) throws ProcessingException {
    final JdbcAccessService jdbcAccessService = getJdbcAccessService();
    final ParameterAccessor paramAccessor = new ParameterAccessor(blackboard, _configuration);
    final ResultCollector resultCollector =
      new ResultCollector(paramAccessor, _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
    if (recordIds != null) {
      for (final String id : recordIds) {
        try {
          paramAccessor.setCurrentRecord(id);
          final String dbUrl = getDbUrl(paramAccessor);
          final AnyMap dbProps = getDbProps(paramAccessor);
          final String stmt = getStatement(paramAccessor);
          final List<Value> stmtValues = getStatementValues(paramAccessor, blackboard.getMetadata(id));
          final SqlExecutor executor =
            jdbcAccessService.executePrepared(dbUrl, dbProps, stmt, 0, new MessageCollector.LogWarn(getClass()));
          final Record result = executor.executeAndMerge(stmtValues);
          if (result != null) {
            blackboard.getMetadata(id).putAll(result.getMetadata());
          }
          resultCollector.addResult(id);
        } catch (final Exception e) {
          resultCollector.addFailedResult(id, e);
        }
      }
    }
    return resultCollector.getResultIds();
  }

  /** @return a {@link JdbcAccessService} service. */
  private synchronized JdbcAccessService getJdbcAccessService() throws ProcessingException {
    if (_jdbcAccess == null) {
      try {
        _jdbcAccess = ServiceUtils.getService(JdbcAccessService.class);
      } catch (final Exception ex) {
        _log.warn("Error while waiting for JdbcAccessService service to come up.", ex);
      }
      if (_jdbcAccess == null) {
        throw new ProcessingException("No JdbcAccessService service available, giving up");
      }
    }
    return _jdbcAccess;
  }
}
