/*******************************************************************************
 * Copyright (c) 2008, 2012 EMPOLIS Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (EMPOLIS Information Management GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.jdbc.internal;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.common.logging.MessageCollector;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Record;

/**
 * helper class that tries to convert a result set to a collection of records.
 * 
 */
public class ResultSetToAnyConverter {

  /**
   * Data types that can be converted to an Any Value will be stored as record metadata, while binary data is appended
   * as an attachment.
   * 
   * The attachments will be kept in memory, so take please care of memory consumption.
   * 
   * When conversion fails, an exception will be thrown.
   * 
   * @param resultSet
   *          the result set to be converted into a collection of records.
   * @return a list of records that have been created by the results contained in the resultSet, null, if resultSet also
   *         was null.
   * @throws SQLException
   */
  public static Collection<Record> convertResultSetToAny(final ResultSet resultSet, final long maxAttachmentSize,
    final MessageCollector messages) throws SQLException, IOException {
    if (resultSet == null) {
      return null;
    }
    final DataFactory df = DataFactory.DEFAULT;
    final Collection<Record> resultRecords = new ArrayList<Record>();
    while (resultSet.next()) {
      final Record record = df.createRecord();
      final AnyMap recordMetadata = record.getMetadata();
      final ResultSetMetaData resultSetMetadata = resultSet.getMetaData();
      // columns start with 1
      for (int i = 1; i <= resultSetMetadata.getColumnCount(); i++) {
        final String columnName = resultSetMetadata.getColumnName(i).toUpperCase(Locale.ENGLISH);
        final int columnType = resultSetMetadata.getColumnType(i);
        switch (columnType) {
          case Types.BIT:
          case Types.BOOLEAN:
            final boolean booleanValue = resultSet.getBoolean(i);
            if (!resultSet.wasNull()) {
              recordMetadata.put(columnName, df.createBooleanValue(booleanValue));
            }
            break;
          case Types.BIGINT:
          case Types.INTEGER:
          case Types.SMALLINT:
          case Types.TINYINT:
            final long longValue = resultSet.getLong(i);
            if (!resultSet.wasNull()) {
              recordMetadata.put(columnName, df.createLongValue(longValue));
            }
            break;
          case Types.DOUBLE:
          case Types.FLOAT:
          case Types.REAL:
            final double doubleValue = resultSet.getDouble(i);
            if (!resultSet.wasNull()) {
              recordMetadata.put(columnName, df.createDoubleValue(doubleValue));
            }
            break;
          case Types.DECIMAL:
          case Types.NUMERIC:
            final BigDecimal bigDecimalValue = resultSet.getBigDecimal(i);
            if (!resultSet.wasNull()) {
              if (bigDecimalValue.scale() > 0) {
                recordMetadata.put(columnName, df.createDoubleValue(bigDecimalValue.doubleValue()));
              } else {
                recordMetadata.put(columnName, df.createLongValue(bigDecimalValue.longValue()));
              }
            }
            break;
          case Types.DATE:
            final Date dateValue = resultSet.getDate(i);
            if (!resultSet.wasNull()) {
              recordMetadata.put(columnName, df.createDateValue(dateValue));
            }
            break;
          case Types.TIME:
            final Time timeValue = resultSet.getTime(i);
            if (!resultSet.wasNull()) {
              recordMetadata.put(columnName, df.createDateTimeValue(timeValue));
            }
            break;
          case Types.TIMESTAMP:
            final Timestamp timeStampValue = resultSet.getTimestamp(i);
            if (!resultSet.wasNull()) {
              recordMetadata.put(columnName, df.createDateTimeValue(timeStampValue));
            }
            break;
          case Types.CHAR:
          case Types.VARCHAR:
          case Types.NCHAR:
          case Types.NVARCHAR:
          case Types.ROWID:
            final String stringValue = resultSet.getString(i);
            if (!resultSet.wasNull()) {
              recordMetadata.put(columnName, df.createStringValue(stringValue));
            }
            break;
          case Types.BINARY:
          case Types.VARBINARY:
            final byte[] byteValue = resultSet.getBytes(i);
            if (!resultSet.wasNull()) {
              if (byteValue.length > maxAttachmentSize) {
                logWarnAttachmentTooLarge(columnName, byteValue.length, maxAttachmentSize, messages);
              } else {
                record.setAttachment(columnName, byteValue);
              }
            }
            break;
          case Types.BLOB:
          case Types.LONGVARBINARY:
            final Blob blob = resultSet.getBlob(i);
            if (!resultSet.wasNull()) {
              if (blob.length() > maxAttachmentSize) {
                logWarnAttachmentTooLarge(columnName, blob.length(), maxAttachmentSize, messages);
              } else {
                final InputStream binaryStream = blob.getBinaryStream();
                record.setAttachment(columnName, IOUtils.toByteArray(binaryStream));
              }
            }
            break;
          case Types.CLOB:
          case Types.NCLOB:
          case Types.LONGVARCHAR:
          case Types.LONGNVARCHAR:
            final Clob clob = resultSet.getClob(i);
            if (!resultSet.wasNull()) {
              if (clob.length() > maxAttachmentSize) {
                logWarnAttachmentTooLarge(columnName, clob.length(), maxAttachmentSize, messages);
              } else {
                final Reader streamReader = clob.getCharacterStream();
                record.setAttachment(columnName, IOUtils.toByteArray(streamReader, "UTF-8"));
              }
            }
            break;
          case Types.NULL:
            // leave metadata empty...
            break;
          case Types.ARRAY:
          case Types.DATALINK:
          case Types.DISTINCT:
          case Types.JAVA_OBJECT:
          case Types.OTHER:
          case Types.REF:
          case Types.SQLXML:
          case Types.STRUCT:
          default:
            // try to do something with it...
            final Object objectValue = resultSet.getObject(i);
            try {
              if (!resultSet.wasNull()) {
                recordMetadata.put(columnName, df.autoConvertValue(objectValue));
              }
            } catch (final InvalidValueTypeException e) {
              // add the raw bytes as attachment...
              record.setAttachment(columnName, resultSet.getBytes(i));
            }
            break;
        }
      }
      resultRecords.add(record);
    }
    return resultRecords;
  }

  private static void logWarnAttachmentTooLarge(final String columnName, final long length,
    final long maxAttachmentSize, final MessageCollector messages) {
    messages.add("No attachment created from a value in column " + columnName + " because its length " + length
      + " exceeded the allowed length " + maxAttachmentSize);

  }
}
