/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.Collection;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Base class for JobManager tests: defines methods for managing jobs and workflows.
 */
public class JobManagerTestBase extends DeclarativeServiceTestCase {

  /** store name. */
  public static final String STORE_NAME = "test-store";

  /** waiting time between invocations to retrieve updated job run data. */
  private static final long WAITING_TIME_FOR_JOB_RUN_DATA = 1000L;

  /** JobRunDataProvider service under test. */
  protected JobRunDataProvider _jobRunDataProvider;

  /** JobRunEngine service under test. */
  protected JobRunEngine _jobRunEngine;

  /** definition persistence to use. */
  protected DefinitionPersistence _defPersistence;

  /** Store service giving access to object store service. */
  protected ObjectStoreService _objectStoreService;

  /** for serializing records to BON/JSON. */
  protected IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _jobRunDataProvider = getService(JobRunDataProvider.class);
    assertNotNull(_jobRunDataProvider);
    _jobRunEngine = getService(JobRunEngine.class);
    assertNotNull(_jobRunEngine);
    _defPersistence = getService(DefinitionPersistence.class);
    assertNotNull(_defPersistence);
    _objectStoreService = getService(ObjectStoreService.class);
    for (final String storeName : _objectStoreService.getStoreNames()) {
      if ("jobmanager".equals(storeName)) {
        _objectStoreService.clearStore(storeName);
      } else {
        _objectStoreService.removeStore(storeName);
      }
    }
    _objectStoreService.ensureStore(STORE_NAME);
  }

  /**
   * utility method to add jobs.
   */
  protected void addJob(final String jobName, final String workflow, final AnyMap parameters) throws Exception {
    addJob(jobName, workflow, parameters, null);
  }

  /**
   * utility method to add jobs.
   */
  protected void addJob(final String jobName, final String workflow, final AnyMap parameters,
    final String jobRunMode) throws Exception {
    final AnyMap jobAny = DataFactory.DEFAULT.createAnyMap();
    jobAny.put("name", jobName);
    jobAny.put("workflow", workflow);
    if (jobRunMode != null) {
      jobAny.getSeq("modes", true).add(jobRunMode);
    }
    jobAny.put("parameters", parameters);
    final JobDefinition jobDef = new JobDefinition(jobAny);
    _defPersistence.addJob(jobDef);
    final JobDefinition checkDef = _defPersistence.getJob(jobName);
    assertNotNull(checkDef);
  }

  /**
   * Utility method to start a job. Asserts the jobId is not null.
   * 
   * @param jobName
   *          The name of the job to start.
   * @return The id of the started job run.
   * @throws Exception
   *           error occurred while starting the job.
   */
  protected String startJob(final String jobName) throws Exception {
    return startJob(jobName, null);
  }

  /**
   * Utility method to start a job with run mode. Asserts the jobId is not null.
   * 
   * @param jobName
   *          The name of the job to start.
   * @param jobRunMode
   *          may be null.
   * @return The id of the started job run.
   * @throws Exception
   *           error occurred while starting the job.
   */
  protected String startJob(final String jobName, final JobRunMode jobRunMode) throws Exception {
    final String jobId = _jobRunEngine.startJob(jobName, jobRunMode);
    assertNotNull(jobId);
    final AnyMap jobRunData = _jobRunDataProvider.getJobRunData(jobName, jobId);
    assertTaskCounters(0, 0, 0, 0, 0, 0, 0, jobRunData);
    return jobId;
  }

  /**
   * assert tasks counter sum in job run data.
   */
  protected void assertTaskCounters(final AnyMap jobRunData) {
    final AnyMap taskData = jobRunData.getMap(JobManagerConstants.TASK_COUNTER);
    final int createdTasks = taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_CREATED_TASKS).intValue();
    final int succeededTasks =
      taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS).intValue();
    final int retriedByWorker =
      taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_WORKER).intValue();
    final int retriedByTtl = taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_TTL).intValue();
    final int failedAfterRetry =
      taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_RETRIED).intValue();
    final int failedByWorker =
      taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_NOT_RETRIED).intValue();
    final int cancelledTasks = taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_CANCELLED_TASKS).intValue();
    final int obsoleteTasks = taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_OBSOLETE_TASKS).intValue();
    assertEquals(createdTasks, succeededTasks + retriedByWorker + retriedByTtl + failedByWorker + failedAfterRetry
      + cancelledTasks + obsoleteTasks);
  }

  /**
   * assert tasks counter in job run data.
   */
  protected void assertTaskCounters(final int createdTasks, final int succeededTasks, final int retriedAfterError,
    final int retriedAfterTimeout, final int failedWithoutRetry, final int failedAfterRetry,
    final int cancelledTasks, final AnyMap jobRunData) {
    final AnyMap taskData = jobRunData.getMap(JobManagerConstants.TASK_COUNTER);
    assertEquals(createdTasks, taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_CREATED_TASKS).intValue());
    assertEquals(succeededTasks, taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS)
      .intValue());
    assertEquals(retriedAfterError, taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_WORKER)
      .intValue());
    assertEquals(retriedAfterTimeout, taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_RETRIED_TASKS_TTL)
      .intValue());
    assertEquals(failedAfterRetry, taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_RETRIED)
      .intValue());
    assertEquals(failedWithoutRetry,
      taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_NOT_RETRIED).intValue());
    assertEquals(cancelledTasks, taskData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_CANCELLED_TASKS)
      .intValue());
  }

  /**
   * Waits for a job to be completed.
   */
  protected void waitForJobRunCompleted(final String jobName, final String jobId, final long maxWaitTime)
    throws Exception {
    final long sleepTime = 500L;
    final long millisStarted = System.currentTimeMillis();
    while (true) {
      final Collection<String> completedIds = _jobRunDataProvider.getCompletedJobRunIds(jobName);
      if (completedIds.contains(jobId)) {
        return;
      }
      assertTrue("Waited too long for job to complete", System.currentTimeMillis() - millisStarted <= maxWaitTime);
      Thread.sleep(sleepTime);
    }
  }

  /**
   * get job run data, assert job id.
   * 
   * @throws InvalidDefinitionException
   */
  protected AnyMap assertJobRunData(final String jobName, final String jobId) throws JobManagerException,
    InvalidDefinitionException {
    AnyMap data = null;
    int count = 0;
    final int maxCount = 3;
    boolean done = false;
    do {
      data = _jobRunDataProvider.getJobRunData(jobName, jobId);
      try {
        if (data == null
          || data.get(JobManagerConstants.DATA_JOB_ID) == null
          || data.get(JobManagerConstants.WORKFLOW_RUN_COUNTER) == null
          || data.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER).get(
            JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS) == null) {
          // ok, data seems to be just gone from zoo keeper into store.
          // let's try that again but with a small delay...
          Thread.sleep(WAITING_TIME_FOR_JOB_RUN_DATA);
          if (++count > maxCount) {
            fail("Too many tries to get job run data.");
          }
        } else {
          done = true;
        }
      } catch (final Exception e) {
        e.printStackTrace();
        ++count;
      }
    } while (!done);
    assertNotNull(data);
    assertEquals(jobId, AccessAny.getStringRequired(data, "jobId"));
    return data;
  }

  /**
   * Asserts that a job run has the status "CANCELLED".
   * 
   * @param jobName
   *          the job name
   * @param jobRunId
   *          the job run id
   * @return job run data
   * @throws JobManagerException
   *           error
   * @throws InvalidDefinitionException
   */
  protected AnyMap assertJobRunCanceled(final String jobName, final String jobRunId) throws JobManagerException,
    InvalidDefinitionException {
    final AnyMap data = assertJobRunData(jobName, jobRunId);
    assertEquals(JobState.CANCELED.name(), AccessAny.getStringRequired(data, JobManagerConstants.DATA_JOB_STATE));
    return data;
  }

  /**
   * Asserts that a job run has the status "FAILED".
   * 
   * @param jobName
   *          the job name
   * @param jobId
   *          the job run id
   * @return an Any describing the JobRun data
   * @throws JobManagerException
   *           error
   * @throws InvalidDefinitionException
   */
  protected AnyMap assertJobRunFailed(final String jobName, final String jobId) throws JobManagerException,
    InvalidDefinitionException {
    final AnyMap data = assertJobRunData(jobName, jobId);
    assertEquals("FAILED", AccessAny.getStringRequired(data, "state"));
    assertTaskCounters(data);
    return data;
  }

  /**
   * asserts that a job is running.
   * 
   * @param jobName
   *          he job name
   * @param jobId
   *          the job run id.
   * @throws JobManagerException
   *           error
   * @throws InvalidDefinitionException
   */
  protected void assertJobRunning(final String jobName, final String jobId) throws JobManagerException,
    InvalidDefinitionException {
    assertJobRunning(jobName, jobId, JobRunMode.STANDARD);
  }

  /**
   * asserts that a job is running in the given run mode.
   */
  protected void assertJobRunning(final String jobName, final String jobId, final JobRunMode mode)
    throws JobManagerException, InvalidDefinitionException {
    final AnyMap data = _jobRunDataProvider.getJobRunData(jobName, jobId);
    assertNotNull(data);
    assertEquals(jobId, AccessAny.getStringRequired(data, "jobId"));
    assertEquals("RUNNING", AccessAny.getStringRequired(data, "state"));
    assertEquals(JobRunMode.STANDARD.name(), data.getStringValue("mode"));
  }

  /**
   * Asserts that a job run has the status "SUCCEEDED".
   * 
   * @param jobName
   *          the job name
   * @param jobId
   *          the job run id
   * @return an Any describing the JobRun data
   * @throws JobManagerException
   *           error
   * @throws InvalidDefinitionException
   */
  protected AnyMap assertJobRunSucceeded(final String jobName, final String jobId) throws JobManagerException,
    InvalidDefinitionException {
    final AnyMap data = assertJobRunData(jobName, jobId);
    assertEquals("SUCCEEDED", AccessAny.getStringRequired(data, "state"));
    // assertTaskCounters(data);
    return data;
  }

  /**
   * Asserts that a job run has the status "SUCCEEDED" with a given number of successful workflow runs.
   * 
   * @param jobName
   *          the job name
   * @param jobId
   *          the job run id
   * @param noOfWorkflowRuns
   *          the expected number of workflow runs
   * @return job run data
   * @throws JobManagerException
   *           error
   * @throws InvalidDefinitionException
   */
  protected AnyMap assertJobRunSucceeded(final String jobName, final String jobId, final int noOfWorkflowRuns)
    throws JobManagerException, InvalidDefinitionException {
    final AnyMap data = assertJobRunSucceeded(jobName, jobId);
    final AnyMap workflowRunData = data.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    assertEquals("jobrun - wrong number of successful workflow runs: " + data.toString(), noOfWorkflowRuns,
      workflowRunData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS).intValue());
    return data;
  }

  /**
   * assert tasks counter in job run data.
   */
  protected void assertWorkflowCounters(final int startedWorkflows, final int activeWorkflows,
    final int successfulWorkflows, final int failedWorkflows, final int canceledWorkflows, final AnyMap jobRunData) {
    final AnyMap workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    assertEquals(startedWorkflows,
      workflowRunData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS).intValue());
    assertEquals(activeWorkflows,
      workflowRunData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_ACTIVE_WORKFLOW_RUNS).intValue());
    assertEquals(successfulWorkflows,
      workflowRunData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS).intValue());
    assertEquals(failedWorkflows,
      workflowRunData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS).intValue());
    assertEquals(canceledWorkflows,
      workflowRunData.getLongValue(JobManagerConstants.DATA_JOB_NO_OF_CANCELED_WORKFLOW_RUNS).intValue());
  }

  /**
   * Creates the job's parameters.
   * 
   * @param workerParameterValue
   *          value for workerParameter
   * @param storeName
   *          store name
   * @param tempStoreName
   *          the name of the temp store
   * @return parameter Any
   * @throws Exception
   *           any exception during creation of Any.
   */
  protected AnyMap createJobParameters(final String workerParameterValue, final String storeName,
    final String tempStoreName) throws Exception {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    parameters.put("store", storeName);
    parameters.put("tempStore", tempStoreName);
    parameters.put("workerParameter", workerParameterValue);
    return parameters;
  }

}
