/*******************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.smila.jobmanager.JobRun;
import org.eclipse.smila.jobmanager.definitions.WorkflowAction;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;

/**
 * Test handling of jobs with barriers.
 */
public class TestBarrierJobs extends JobTaskProcessingTestBase {

  private static final String RESULT_STORE = "store";

  private static final String INPUT_BUCKET = "inBucket";

  private TaskManager _taskManager;

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _objectStoreService.ensureStore(RESULT_STORE);
    _objectStoreService.clearStore(RESULT_STORE);
    _taskManager = getService(TaskManager.class);
  }

  /** test if a job with a single barrier is compiled correctly. */
  public void testPrepareBarrierJob() throws Exception {
    final String jobName = "barrierJob";
    _objectStoreService.putObject(RESULT_STORE, INPUT_BUCKET + "/input", "processMe".getBytes());
    final String jobRunId = _jobRunEngine.startJob(jobName);
    try {
      final JobRun jobRun = _jobRunEngine.ensureJobRun(jobName, jobRunId);
      assertTrue(jobRun.hasBarriers());

      final WorkflowAction expectedBarrier = jobRun.getAction(1);

      assertEquals(1, jobRun.getBarrierActions().size());
      assertSame(expectedBarrier, jobRun.getBarrierActions().iterator().next());

      assertNull(jobRun.getPrecedingBarriers(expectedBarrier));

      final Collection<WorkflowAction> barriers = jobRun.getBarriersForAction(jobRun.getAction(0));
      assertEquals(1, barriers.size());
      assertSame(expectedBarrier, barriers.iterator().next());

      assertNull(jobRun.getBarriersForAction(expectedBarrier));
    } finally {
      _jobRunEngine.cancelJob(jobName, jobRunId);
    }
  }

  public void testRunBarrierJob() throws Exception {
    final String jobName = "barrierJob";
    _objectStoreService.putObject(RESULT_STORE, INPUT_BUCKET + "/input1", "processMe".getBytes());
    _objectStoreService.putObject(RESULT_STORE, INPUT_BUCKET + "/input2", "processMeToo".getBytes());
    _objectStoreService.putObject(RESULT_STORE, INPUT_BUCKET + "/input3", "dontForgetMe".getBytes());
    final String jobRunId = _jobRunEngine.startJob(jobName);
    try {
      for (int i = 0; i < 3; i++) {
        final Task task = _taskManager.getTask("triggeredWorker", null);
        assertNotNull(task);
        createBulk(task, "output", "triggeredWorker" + i);
        final List<Task> barrierTasks = getNextTasks(task, successResult());
        if (i < 2) {
          assertTrue(barrierTasks.isEmpty());
        } else {
          assertNull(_taskManager.getTask("triggeredWorker", null));
          assertEquals(3, barrierTasks.size());
          for (final Task barrierTask : barrierTasks) {
            assertEquals("barrierWorker", barrierTask.getWorkerName());
            createBulk(barrierTask, "output", "barrier result");
            finishFinalTask(barrierTask, successResult());
          }
        }
      }
      assertNull(_jobRunDataProvider.getJobRunInfo(jobName));
      assertEquals("SUCCEEDED", _jobRunDataProvider.getJobRunData(jobName, jobRunId).getStringValue("state"));
    } finally {
      if (_jobRunDataProvider.getJobRunInfo(jobName) != null) {
        _jobRunEngine.cancelJob(jobName, jobRunId);
      }
    }
  }

  /** test if a job with two barriers is compiled correctly. */
  public void testPrepareTwoBarriersJob() throws Exception {
    final String jobName = "twoBarriersJob";
    _objectStoreService.putObject(RESULT_STORE, INPUT_BUCKET + "/input", "processMe".getBytes());
    final String jobRunId = _jobRunEngine.startJob(jobName);
    try {
      final JobRun jobRun = _jobRunEngine.ensureJobRun(jobName, jobRunId);
      assertTrue(jobRun.hasBarriers());

      final WorkflowAction expectedBarrier1 = jobRun.getAction(1);
      final WorkflowAction expectedBarrier2 = jobRun.getAction(3);

      assertEquals(2, jobRun.getBarrierActions().size());
      final Iterator<WorkflowAction> barrierActions = jobRun.getBarrierActions().iterator();
      assertSame(expectedBarrier1, barrierActions.next());
      assertSame(expectedBarrier2, barrierActions.next());

      assertNull(jobRun.getPrecedingBarriers(expectedBarrier1));
      final Collection<WorkflowAction> precedingBarriers = jobRun.getPrecedingBarriers(expectedBarrier2);
      assertEquals(1, precedingBarriers.size());
      assertSame(expectedBarrier1, precedingBarriers.iterator().next());

      Collection<WorkflowAction> barriers = jobRun.getBarriersForAction(jobRun.getAction(0));
      assertEquals(2, barriers.size());
      assertTrue(barriers.contains(expectedBarrier1));
      assertTrue(barriers.contains(expectedBarrier2));

      barriers = jobRun.getBarriersForAction(expectedBarrier1);
      assertEquals(1, barriers.size());
      assertTrue(barriers.contains(expectedBarrier2));

      barriers = jobRun.getBarriersForAction(jobRun.getAction(2));
      assertEquals(1, barriers.size());
      assertTrue(barriers.contains(expectedBarrier2));

      assertNull(jobRun.getBarriersForAction(expectedBarrier2));
    } finally {
      _jobRunEngine.cancelJob(jobName, jobRunId);
    }
  }

  public void testRunTwoBarriersJob() throws Exception {
    final String jobName = "twoBarriersJob";
    _objectStoreService.putObject(RESULT_STORE, INPUT_BUCKET + "/input1", "processMe".getBytes());
    _objectStoreService.putObject(RESULT_STORE, INPUT_BUCKET + "/input2", "processMeToo".getBytes());
    final String jobRunId = _jobRunEngine.startJob(jobName);
    try {
      final List<Task> stage2Tasks = new ArrayList<>();
      for (int i = 0; i < 2; i++) {
        final Task task = _taskManager.getTask("triggeredWorker", null);
        assertNotNull(task);
        createBulk(task, "output", "triggeredWorker" + i);
        final List<Task> barrierTasks = getNextTasks(task, successResult());
        if (i == 0) {
          assertTrue(barrierTasks.isEmpty());
        } else {
          assertNull(_taskManager.getTask("triggeredWorker", null));
          assertEquals(2, barrierTasks.size());
          for (final Task barrierTask : barrierTasks) {
            assertEquals("barrierWorker", barrierTask.getWorkerName());
            createBulk(barrierTask, "output", "barrier result");
            stage2Tasks.addAll(getNextTasks(barrierTask, successResult()));
          }
        }
      }
      assertEquals(2, stage2Tasks.size());
      for (int i = 0; i < 2; i++) {
        final Task task = stage2Tasks.get(i);
        assertNotNull(task);
        assertEquals("intermediateWorker", task.getWorkerName());
        createBulk(task, "output", "intermediateWorker" + i);
        final List<Task> barrierTasks = getNextTasks(task, successResult());
        if (i == 0) {
          assertTrue(barrierTasks.isEmpty());
        } else {
          assertEquals(2, barrierTasks.size());
          for (final Task barrierTask : barrierTasks) {
            assertEquals("barrierWorker", barrierTask.getWorkerName());
            createBulk(barrierTask, "output", "barrier result");
            finishFinalTask(barrierTask, successResult());
          }
        }
      }
      assertNull(_jobRunDataProvider.getJobRunInfo(jobName));
      assertEquals("SUCCEEDED", _jobRunDataProvider.getJobRunData(jobName, jobRunId).getStringValue("state"));
    } finally {
      if (_jobRunDataProvider.getJobRunInfo(jobName) != null) {
        _jobRunEngine.cancelJob(jobName, jobRunId);
      }
    }
  }
}
