/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.io.IOException;

import junit.framework.TestCase;

import org.eclipse.smila.jobmanager.exceptions.ConfigNotFoundException;
import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.InvalidConfigException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;
import org.eclipse.smila.jobmanager.persistence.RunStorageException;
import org.eclipse.smila.objectstore.BadRequestException;
import org.eclipse.smila.objectstore.InvalidStoreNameException;
import org.eclipse.smila.objectstore.NoSuchObjectException;
import org.eclipse.smila.objectstore.NoSuchStoreException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ServiceUnavailableException;
import org.eclipse.smila.objectstore.StoreExistsException;

/**
 * test some basic JobManagerException logic.
 */
public class TestJobManagerException extends TestCase {

  /** check if recoverable flag works correctly. */
  public void testRecoverableFlag() {
    assertTrue(new JobManagerException("custom", true).isRecoverable());
    assertFalse(new JobManagerException("custom", false).isRecoverable());
  }

  /** check if recoverable flag is detected correctly from causes. */
  public void testRecoverableExceptionRecognition() {
    assertTrue(new JobManagerException("io", new IOException()).isRecoverable());
    assertTrue(new JobManagerException("unavailable", new ServiceUnavailableException("ObjectStoreService"))
      .isRecoverable());
    assertFalse(new JobManagerException("objectstore", new ObjectStoreException("ObjectStoreService"))
      .isRecoverable());
    assertFalse(new JobManagerException("badrequesst", new BadRequestException("ObjectStoreService"))
      .isRecoverable());
    assertFalse(new JobManagerException("invalidstorename", new InvalidStoreNameException("ObjectStoreService"))
      .isRecoverable());
    assertFalse(new JobManagerException("storeexists", new StoreExistsException("ObjectStoreService"))
      .isRecoverable());
    assertFalse(new JobManagerException("nosuchstore", new NoSuchStoreException("ObjectStoreService"))
      .isRecoverable());
    assertFalse(new JobManagerException("nosuchobject", new NoSuchObjectException("ObjectStoreService"))
      .isRecoverable());
    assertFalse(new JobManagerException("runtime", new NullPointerException()).isRecoverable());
    assertFalse(new JobManagerException("anything", new OutOfMemoryError()).isRecoverable());
  }

  /** check if recoverable flag is copied correctly from causing JobMgrException.. */
  public void testRecoverableFlagInheritance() {
    assertTrue(new JobManagerException("cascaded", new JobManagerException("custom", true)).isRecoverable());
    assertFalse(new JobManagerException("cascaded", new JobManagerException("custom", false)).isRecoverable());
  }

  /** check sub classes of JobManagerExceptions. */
  public void testSubExceptions() {
    assertFalse(new IllegalJobStateException("jobstate").isRecoverable());
    assertFalse(new InvalidConfigException("invalid").isRecoverable());
    assertFalse(new InvalidConfigException("invalid", new IOException()).isRecoverable());
    assertFalse(new ConfigNotFoundException("notfound").isRecoverable());
    assertFalse(new PersistenceException("persistence", new ObjectStoreException("server")).isRecoverable());
    assertFalse(new PersistenceException("persistence", false).isRecoverable());
    assertTrue(new PersistenceException("persistence", true).isRecoverable());
    assertTrue(new PersistenceException("persistence", new IOException()).isRecoverable());
    assertTrue(new PersistenceException("persistence", new IOException(), true).isRecoverable());
    assertFalse(new PersistenceException("persistence", new IOException(), false).isRecoverable());
    assertFalse(new RunStorageException("runstorage", false).isRecoverable());
    assertTrue(new RunStorageException("runstorage", true).isRecoverable());
    assertTrue(new RunStorageException("runstorage", new IOException(), true).isRecoverable());
    assertFalse(new RunStorageException("runstorage", new IOException(), false).isRecoverable());
  }
}
