/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test;

import java.util.Collection;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;

/** Test that JobManager starts a completing workflow run if needed. */
public class TestJobManagerJobCompletingWorkflowRun extends JobTaskProcessingTestBase {

  /** worker name for initial worker (startAction worker). */
  private static final String START_ACTION_WORKER = "virtualCompletingTasksWorker";

  /** workflow name. */
  private static final String WORKFLOW_NAME = "virtualCompletingTasksWorkflow";

  /** parameter for store. */
  private static final String KEY_STORE_NAME = "store";

  /** The object store service. */
  private ObjectStoreService _defStorage;

  /** the jobId (to be able to finish it when test is finished. */
  private String _jobId;

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _defStorage = getService(ObjectStoreService.class);
    for (final String storeName : _objectStoreService.getStoreNames()) {
      if ("jobmanager".equals(storeName)) {
        _objectStoreService.clearStore(storeName);
      } else {
        _objectStoreService.removeStore(storeName);
      }
    }
    _defStorage.ensureStore(getName());

    // default job
    final AnyMap jobAny = AccessAny.FACTORY.createAnyMap();
    jobAny.put("name", getName());
    final AnyMap parametersAny = AccessAny.FACTORY.createAnyMap();
    parametersAny.put(KEY_STORE_NAME, getName());
    jobAny.put("parameters", parametersAny);
    jobAny.put("workflow", WORKFLOW_NAME);
    _defPersistence.addJob(new JobDefinition(jobAny));

    _jobId = _jobRunEngine.startJob(getName());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
    _defPersistence.removeJob(getName());
  }

  /**
   * Tests finishTask for a workflow that has no following tasks for this worker because an optional bulk has not been
   * created.
   * 
   * @throws Exception
   *           an error occurred while creating definitions or during job management.
   */
  public void testSwitchJobStateToCompleting() throws Exception {
    final Task initialTask = _jobTaskProcessor.getInitialTask(START_ACTION_WORKER, getName());
    assertNotNull(initialTask);
    assertFalse(initialTask.getParameters().containsKey(Task.PROPERTY_IS_COMPLETING_TASK));
    final ResultDescription resultDescription =
      new ResultDescription(TaskCompletionStatus.SUCCESSFUL, null, null, null);
    // do not create output bulks
    final Collection<Task> followUpTasks = getNextTasks(initialTask, resultDescription);
    assertTrue("Follow up tasks should be empty.", followUpTasks.isEmpty());
    _jobRunEngine.finishJob(getName(), _jobId);
    // check the job's status, must be completing
    JobRunInfo jobRunInfo = _jobRunDataProvider.getJobRunInfo(getName());
    assertEquals(JobState.COMPLETING, jobRunInfo.getState());

    // get the completing task
    final Task completingTask = _taskStorage.getTask(START_ACTION_WORKER, null, null);
    // be sure we get a completing task.
    assertTrue(completingTask.getProperties().containsKey(Task.PROPERTY_IS_COMPLETING_TASK));
    // check that after finishing this task the job run info will no longer be present
    finishFinalTask(completingTask, resultDescription);
    jobRunInfo = _jobRunDataProvider.getJobRunInfo(getName());
    // it must be gone now!
    assertNull(jobRunInfo);
  }

}
