/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test.definitions;

import junit.framework.TestCase;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.definitions.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.definitions.DataObjectTypeDefinition.Mode;

/**
 * Tests for DataObjectTypeDefinitio class.
 * 
 * Other tests are covered by TestDefinitionPersistence.testReadPredefinedDataObjectTypeDefinitions().
 */
public class TestDataObjectTypeDefinition extends TestCase {

  /**
   * tests the conversion of a data object type definition from and to Any.
   * 
   * @throws Exception
   *           error
   */
  public void testToAny() throws Exception {
    final AnyMap dotAny = AccessAny.FACTORY.createAnyMap();
    dotAny.put("name", "myName");

    final AnyMap persistentModeDef = AccessAny.FACTORY.createAnyMap();
    persistentModeDef.put("store", "myPersStore");
    persistentModeDef.put("object", "myPersObject");
    dotAny.put(Mode.PERSISTENT.toString(), persistentModeDef);

    final AnyMap transientModeDef = AccessAny.FACTORY.createAnyMap();
    transientModeDef.put("store", "myTransStore");
    transientModeDef.put("object", "myTransObject");
    dotAny.put(Mode.TRANSIENT.toString(), transientModeDef);

    final DataObjectTypeDefinition dot = new DataObjectTypeDefinition(dotAny);
    final AnyMap dotNewAny = dot.toAny();
    final DataObjectTypeDefinition dotNew = new DataObjectTypeDefinition(dotNewAny);

    assertEquals(dotNew.getName(), "myName");
    assertEquals(dotNew.getStore(Mode.PERSISTENT).getExpression(), "myPersStore");
    assertEquals(dotNew.getObject(Mode.PERSISTENT).getExpression(), "myPersObject");
    assertEquals(dotNew.getStore(Mode.TRANSIENT).getExpression(), "myTransStore");
    assertEquals(dotNew.getObject(Mode.TRANSIENT).getExpression(), "myTransObject");
    assertTrue(dotNew.hasPersistentMode());
    assertTrue(dotNew.hasTransientMode());
  }

  /**
   * Tests if the correct exception is thrown if no name is set.
   * 
   * @throws Exception
   *           exception if something goes wrong during creating data object type definition
   */
  public void testNoName() throws Exception {
    final AnyMap dotAny = AccessAny.FACTORY.createAnyMap();
    try {
      new DataObjectTypeDefinition(dotAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests if the correct exception is thrown if an invalid name is set.
   * 
   * @throws Exception
   *           exception if something goes wrong during creating data object type definition
   */
  public void testInvalidName() throws Exception {
    final AnyMap dotAny = AccessAny.FACTORY.createAnyMap();
    dotAny.put("name", "+/dot");
    try {
      new DataObjectTypeDefinition(dotAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests if the correct exception is thrown if no data object type is set.
   * 
   * @throws Exception
   *           exception if something goes wrong during creating data object type definition
   */
  public void testNoObject() throws Exception {
    final AnyMap dotAny = AccessAny.FACTORY.createAnyMap();
    dotAny.put(DataObjectTypeDefinition.KEY_NAME, "name");
    try {
      new DataObjectTypeDefinition(dotAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }
}
