/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test.definitions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import junit.framework.TestCase;

import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;

/**
 * Tests for validation in job parsing.
 * 
 */
public class TestJobDefinition extends TestCase {
  /**
   * Tests a job with missing "name". Must fail.
   * 
   * @throws Exception
   *           Exception when converting to or from Any.
   */
  public void testMissingName() throws Exception {
    final AnyMap jobAny = DataFactory.DEFAULT.createAnyMap();
    jobAny.put("job-name", "job");
    jobAny.put("workflow", "workflow");
    try {
      new JobDefinition(jobAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests a job with invalid "name". Must fail.
   * 
   * @throws Exception
   *           Exception when converting to or from Any.
   */
  public void testInvalidName() throws Exception {
    final AnyMap jobAny = DataFactory.DEFAULT.createAnyMap();
    jobAny.put("name", "job+");
    jobAny.put("workflow", "workflow");
    try {
      new JobDefinition(jobAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests job without mandatory "workflow" name.
   * 
   * @throws Exception
   *           Exception when converting to or from Any.
   */
  public void testMissingWorkflow() throws Exception {
    final AnyMap jobAny = DataFactory.DEFAULT.createAnyMap();
    jobAny.put("name", "job");
    final AnyMap parametersAny = DataFactory.DEFAULT.createAnyMap();
    parametersAny.put("index-name", "index");
    jobAny.put("parameters", parametersAny);
    try {
      new JobDefinition(jobAny);
      fail("should not work");
    } catch (final Exception ex) {
      assertTrue("wrong exception caught: " + ex.toString(), ex instanceof InvalidDefinitionException);
    }
  }

  /**
   * Tests a working job definition. Must not fail.
   * 
   * @throws Exception
   *           Exception when converting to or from Any.
   */
  public void testWorkingDefinition() throws Exception {
    final AnyMap jobAny = createValidJob("job");
    final JobDefinition jdef = new JobDefinition(jobAny);
    assertEquals("name does not match", jobAny.getStringValue("name"), jdef.getName());
    assertEquals("workflow does not match", jobAny.getStringValue("workflow"), jdef.getWorkflow());
    assertEquals("parameters do not match", jobAny.getMap("parameters").getStringValue("index-name"), jdef
      .getParameters().getStringValue("index-name"));
    assertNull(jdef.getJobRunModes());
    assertNull(jdef.getDefaultJobRunMode());
  }

  /**
   * Tests setting a job run mode in a working job definition. Must not fail.
   */
  public void testSetJobRunMode() throws Exception {
    final AnyMap jobAny = createValidJob("job");
    final AnySeq modes = jobAny.getSeq("modes", true);
    modes.add("runOnce");
    final JobDefinition jdef = new JobDefinition(jobAny);
    assertEquals("name does not match", jobAny.getStringValue("name"), jdef.getName());
    assertEquals("workflow does not match", jobAny.getStringValue("workflow"), jdef.getWorkflow());
    assertEquals("parameters do not match", jobAny.getMap("parameters").getStringValue("index-name"), jdef
      .getParameters().getStringValue("index-name"));
    assertEquals(Arrays.asList(JobRunMode.RUNONCE), jdef.getJobRunModes());
    assertEquals(JobRunMode.RUNONCE, jdef.getDefaultJobRunMode());
  }

  /**
   * Tests setting two job run modes in a working job definition. Must not fail.
   */
  public void testSetJobRunModes() throws Exception {
    final AnyMap jobAny = createValidJob("job");
    final AnySeq modes = jobAny.getSeq("modes", true);
    modes.add("runOnce");
    modes.add("standard");
    final JobDefinition jdef = new JobDefinition(jobAny);
    assertEquals("name does not match", jobAny.getStringValue("name"), jdef.getName());
    assertEquals("workflow does not match", jobAny.getStringValue("workflow"), jdef.getWorkflow());
    assertEquals("parameters do not match", jobAny.getMap("parameters").getStringValue("index-name"), jdef
      .getParameters().getStringValue("index-name"));
    assertEquals(Arrays.asList(JobRunMode.RUNONCE, JobRunMode.STANDARD), jdef.getJobRunModes());
    assertEquals(JobRunMode.RUNONCE, jdef.getDefaultJobRunMode());
    assertTrue(jdef.getTaskDelayWorkers().isEmpty());
  }

  /**
   * Tests setting an invalid job run mode in a otherwise working job definition. Must fail.
   */
  public void testSetInvalidJobRunMode() throws Exception {
    final AnyMap jobAny = createValidJob("job");
    final AnySeq modes = jobAny.getSeq("modes", true);
    modes.add("doesNotRun");
    try {
      new JobDefinition(jobAny);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK;
    }
  }

  /** test task control configuration with a single worker. */
  public void testTaskControlForOneWorker() throws Exception {
    final AnyMap job = createValidJob("jobTaskControl");
    job.getMap("taskControl", true).getSeq("workers", true).add("throttleWorker");
    job.getMap("taskControl", true).put("delay", 1234);
    final JobDefinition jobDef = new JobDefinition(job);
    assertEquals(1, jobDef.getTaskDelayWorkers().size());
    assertEquals("throttleWorker", jobDef.getTaskDelayWorkers().get(0));
    assertEquals(1234, jobDef.getTaskDelay());
  }

  /** test task control configuration with two workers. */
  public void testTaskControlForTwoWorkers() throws Exception {
    final AnyMap job = createValidJob("jobTaskControl");
    job.getMap("taskControl", true).getSeq("workers", true).add("throttleWorker1");
    job.getMap("taskControl", true).getSeq("workers", true).add("throttleWorker2");
    job.getMap("taskControl", true).put("delay", 2358);
    final JobDefinition jobDef = new JobDefinition(job);
    assertEquals(2, jobDef.getTaskDelayWorkers().size());
    assertEquals("throttleWorker1", jobDef.getTaskDelayWorkers().get(0));
    assertEquals("throttleWorker2", jobDef.getTaskDelayWorkers().get(1));
    assertEquals(2358, jobDef.getTaskDelay());
  }

  /** test invalid task control config: no workers added. */
  public void testTaskControlWithoutWorkers() throws Exception {
    final AnyMap job = createValidJob("jobTaskControl");
    job.getMap("taskControl", true).put("delay", 2358);
    try {
      new JobDefinition(job);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK;
    }
    job.getMap("taskControl", true).getSeq("workers", true);
    try {
      new JobDefinition(job);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK;
    }
  }

  /** test invalid task control config: no delay specified. */
  public void testTaskControlWithoutDelay() throws Exception {
    final AnyMap job = createValidJob("jobTaskControl");
    job.getMap("taskControl", true).getSeq("workers", true).add("throttleWorker1");
    job.getMap("taskControl", true).getSeq("workers", true).add("throttleWorker2");
    try {
      new JobDefinition(job);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK;
    }
  }

  /** test invalid task control config: no delay specified. */
  public void testTaskControlWithInvalidDelay() throws Exception {
    final AnyMap job = createValidJob("jobTaskControl");
    job.getMap("taskControl", true).getSeq("workers", true).add("throttleWorker1");
    job.getMap("taskControl", true).getSeq("workers", true).add("throttleWorker2");
    job.getMap("taskControl", true).put("delay", "one minute");
    try {
      new JobDefinition(job);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK;
    }
    job.getMap("taskControl", true).put("delay", -1);
    try {
      new JobDefinition(job);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK;
    }
  }

  /** test task control configuration with two workers. */
  public void testTaskControlToAny() throws Exception {
    final AnyMap job = createValidJob("jobTaskControl");
    job.getMap("taskControl", true).getSeq("workers", true).add("throttleWorker1");
    job.getMap("taskControl", true).getSeq("workers", true).add("throttleWorker2");
    job.getMap("taskControl", true).put("delay", 3579);
    job.getMap("taskControl", true).put("description", "don't work too fast");
    final JobDefinition jobDef = new JobDefinition(job);
    final AnyMap basicDef = jobDef.toAny(false);
    assertTaskControlSection(basicDef);
    assertFalse(basicDef.getMap("taskControl").containsKey("description"));
    final AnyMap fullDef = jobDef.toAny(true);
    assertTaskControlSection(fullDef);
    assertTrue(fullDef.getMap("taskControl").containsKey("description"));
  }

  private void assertTaskControlSection(final AnyMap basicDef) {
    assertTrue(basicDef.containsKey("taskControl"));
    assertTrue(basicDef.getMap("taskControl").containsKey("workers"));
    assertEquals(2, basicDef.getMap("taskControl").getSeq("workers").size());
    assertEquals("throttleWorker1", basicDef.getMap("taskControl").getSeq("workers").getStringValue(0));
    assertEquals("throttleWorker2", basicDef.getMap("taskControl").getSeq("workers").getStringValue(1));
    assertTrue(basicDef.getMap("taskControl").containsKey("delay"));
    assertEquals(3579, basicDef.getMap("taskControl").getLongValue("delay").intValue());
  }

  /**
   * Tests jobs definition list.
   * 
   * @throws Exception
   *           Exception when converting to or from Any.
   */
  public void testListOfDefinition() throws Exception {
    final AnyMap jobsAny = DataFactory.DEFAULT.createAnyMap();
    final AnySeq jobsSeq = DataFactory.DEFAULT.createAnySeq();
    final AnyMap job1 = createValidJob("job1");
    final AnyMap job2 = createValidJob("job2");
    jobsSeq.add(job1);
    jobsSeq.add(job2);
    jobsAny.put("jobs", jobsSeq);
    final Collection<JobDefinition> parseJobs = new ArrayList<JobDefinition>();
    for (final Any any : jobsAny.getSeq(JobDefinition.KEY_JOBS)) {
      parseJobs.add(JobDefinition.parseJob(any.asMap()));
    }
    assertEquals(2, parseJobs.size());
    for (final JobDefinition job : parseJobs) {
      assertTrue("job1".equals(job.getName()) || "job2".equals(job.getName()));
    }
  }

  /**
   * Tests the conversion to and from Any.
   * 
   * @throws Exception
   *           Exception when converting to or from Any.
   */
  public void testWorkingDefinitionToAndFromAny() throws Exception {
    final AnyMap jobAny = DataFactory.DEFAULT.createAnyMap();
    jobAny.put("name", "job");
    final AnySeq modesAny = jobAny.getSeq("modes", true);
    modesAny.add("runOnce");
    final AnyMap parametersAny = jobAny.getMap("parameters", true);
    parametersAny.put("index-name", "index");
    final AnyMap mapInMap = parametersAny.getMap("mapInMap", true);
    mapInMap.put("anotherVariable", "${index-name}");
    jobAny.put("workflow", "workflow");
    jobAny.put("additionalData", "additional data");
    final JobDefinition jdef = new JobDefinition(jobAny);
    final AnyMap reEngineeredAny = jdef.toAny(true);
    assertEquals(jobAny, reEngineeredAny);
    jobAny.remove("additionalData");
    final AnyMap anyWithoutDetails = jdef.toAny(false);
    assertEquals(jobAny, anyWithoutDetails);

  }

  /** create job as AnyMap representation for testing. */
  private AnyMap createValidJob(final String name) {
    final AnyMap jobAny = DataFactory.DEFAULT.createAnyMap();
    jobAny.put("name", name);
    final AnyMap parametersAny = DataFactory.DEFAULT.createAnyMap();
    parametersAny.put("index-name", "index");
    jobAny.put("parameters", parametersAny);
    jobAny.put("workflow", "workflow");
    jobAny.put("additionalData", "additional data");
    return jobAny;
  }
}
