/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Schank (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.test.httphandler;

import java.util.Date;

import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.util.RequestHandler;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.exceptions.ConfigNotFoundException;
import org.eclipse.smila.jobmanager.httphandler.WorkflowHandler;
import org.eclipse.smila.jobmanager.httphandler.WorkflowsHandler;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Test cases for the {@link WorkflowHandler} and {@link WorkflowsHandler}.
 */
public class TestWorkflowHandler extends DeclarativeServiceTestCase {

  private WorkflowHandler _workflowHandler;

  private WorkflowsHandler _workflowsHandler;

  /** {@inheritDoc} */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    RequestHandler service = getService(RequestHandler.class, "(uriPattern=/jobmanager/workflows/\\([^/]+\\)/?$)");
    assertTrue(service instanceof WorkflowHandler);
    _workflowHandler = (WorkflowHandler) service;
    service = getService(RequestHandler.class, "(uriPattern=/jobmanager/workflows/?$)");
    assertTrue(service instanceof WorkflowsHandler);
    _workflowsHandler = (WorkflowsHandler) service;
  }

  /** {@inheritDoc} */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /**
   * tests adding, getting and deleting a workflow via the request handler.
   * 
   * @throws Exception
   */
  public void testAddAndGetAndDeleteWorkflow() throws Exception {
    final String workflowBaseUrlPart = _workflowHandler.getRootContextPath() + "/jobmanager/workflows/";
    final String workflowName = "triggeredWorkflow2";
    final Record workflow = DataFactory.DEFAULT.createRecord();
    workflow.getMetadata().put("name", workflowName);
    final AnyMap startAction = DataFactory.DEFAULT.createAnyMap();
    startAction.put("worker", "triggeredWorker");
    final AnyMap input = DataFactory.DEFAULT.createAnyMap();
    input.put("input", "finalBucket");
    final AnyMap output = DataFactory.DEFAULT.createAnyMap();
    output.put("output", "triggeredBucket");
    startAction.put("input", input);
    startAction.put("output", output);
    workflow.getMetadata().put("startAction", startAction);
    workflow.getMetadata().put("additionalData", "some additional data.");
    // this must be removed when inserted via handler.
    workflow.getMetadata().put(DefinitionBase.KEY_READ_ONLY, true);

    // create a workflow
    final Date startDate = new Date();
    final AnyMap result = (AnyMap) _workflowsHandler.process("POST", workflowBaseUrlPart, workflow, null);
    assertEquals(workflow.getMetadata().get("name").toString(), result.get("name").toString());
    // get the complete workflow
    final Record returnDetails = DataFactory.DEFAULT.createRecord();
    returnDetails.getMetadata().put(JobManagerConstants.KEY_RETURN_DETAILS, true);
    AnyMap workflowAny =
      (AnyMap) _workflowHandler.process("GET", workflowBaseUrlPart + workflowName, returnDetails);
    final Date endDate = new Date();
    // check the timestamp of the result.
    final Date dateFromWorkflow = workflowAny.getDateTimeValue(DefinitionBase.KEY_TIMESTAMP);
    assertEquals(result.getDateTimeValue(DefinitionBase.KEY_TIMESTAMP), dateFromWorkflow);
    assertTrue(startDate.compareTo(dateFromWorkflow) <= 0);
    assertTrue(endDate.compareTo(dateFromWorkflow) >= 0);
    // be sure that the readonly flag has been removed from the input any.
    assertFalse(workflowAny.containsKey(DefinitionBase.KEY_READ_ONLY));
    // we must remove timestamp and readOnly flag for comparison...
    final Any timestampAny = workflow.getMetadata().get(JobManagerConstants.TIMESTAMP);
    assertEquals(timestampAny.asValue().asDateTime(), dateFromWorkflow);
    workflow.getMetadata().remove(DefinitionBase.KEY_TIMESTAMP);
    workflow.getMetadata().remove(DefinitionBase.KEY_READ_ONLY);
    workflowAny.remove(JobManagerConstants.TIMESTAMP);
    assertEquals(workflow.getMetadata(), workflowAny);
    // get only the relevant part
    workflowAny = (AnyMap) _workflowHandler.process("GET", workflowBaseUrlPart + workflowName, null);
    final AnyMap reducedAny = DataFactory.DEFAULT.cloneAnyMap(workflow.getMetadata());
    reducedAny.remove("additionalData");
    // re-set the timestamp to be able to check it (we removed it above)
    reducedAny.put(DefinitionBase.KEY_TIMESTAMP, timestampAny);
    assertEquals(reducedAny, workflowAny);
    // remove the workflow
    _workflowHandler.process("DELETE", workflowBaseUrlPart + workflowName, null);

    // check if it really got removed...
    try {
      _workflowHandler.process("GET", workflowBaseUrlPart + workflowName, null);
      fail("ConfigNotFoundException expected.");
    } catch (final ConfigNotFoundException e) {
      ;// ignore
    }
  }

  /** check if a workflow with an invalid name is rejected. */
  public void testAddWorkflowWithInvalidName() throws Exception {
    try {
      final Record workflow = DataFactory.DEFAULT.createRecord();
      workflow.getMetadata().put("name", "this/name+is=invalid?");
      _workflowsHandler.process("POST", "jobmanager/workflows", workflow, null);
      fail("should not work");
    } catch (final InvalidDefinitionException ex) {
      ; // OK
    }
  }
}
