/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.definitions;

import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.common.definitions.ValueExpression;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;

/**
 * Container class for data object type definition elements.
 */
public class DataObjectTypeDefinition extends DefinitionBase {

  /**
   * Mode.
   */
  public enum Mode {
    /** . */
    TRANSIENT("transient"), PERSISTENT("persistent");

    /** . */
    private final String _name;

    /**
     * @param n
     *          mode name
     */
    Mode(final String n) {
      _name = n;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
      return _name;
    }
  };

  /** Property for data object types in json. */
  public static final String KEY_DATA_OBJECT_TYPE_DEFINITIONS = "dataObjectTypes";

  /** The data for the mode definition(s). */
  private final Map<Mode, DataObjectTypeModeDefinition> _modeDefinitions;

  /**
   * Constructor for DataObjectTypeDefinition.
   * 
   * @param dotAny
   *          The any Object with one dataObjectTypeDefinition
   * @throws InvalidDefinitionException
   *           exception if the any object is not filled with all desired values
   */
  public DataObjectTypeDefinition(final AnyMap dotAny) throws InvalidDefinitionException {
    super(dotAny);
    _modeDefinitions = new LinkedHashMap<DataObjectTypeDefinition.Mode, DataObjectTypeModeDefinition>(2);
    final AnyMap persDefAny = (AnyMap) AccessAny.get(dotAny, Mode.PERSISTENT.toString(), ValueType.MAP);
    if (persDefAny != null) {
      _modeDefinitions.put(Mode.PERSISTENT, new DataObjectTypeModeDefinition(persDefAny));
    }
    final AnyMap transDefAny = (AnyMap) AccessAny.get(dotAny, Mode.TRANSIENT.toString(), ValueType.MAP);
    if (transDefAny != null) {
      _modeDefinitions.put(Mode.TRANSIENT, new DataObjectTypeModeDefinition(transDefAny));
    }
    if (persDefAny == null && transDefAny == null) {
      throw new InvalidDefinitionException("You have to set at least one of 'transient' or 'persistent' mode");
    }
  }

  /**
   * Returns this object as an Any representation.
   * 
   * @return Any object describing this object
   */
  @Override
  public AnyMap toAny(final boolean includingAdditionalAttributes) {
    try {
      final AnyMap dotAny = super.toAny(includingAdditionalAttributes);
      if (!includingAdditionalAttributes) {
        if (_modeDefinitions != null && !_modeDefinitions.isEmpty()) {
          for (final Mode mode : _modeDefinitions.keySet()) {
            final DataObjectTypeModeDefinition modeDef = _modeDefinitions.get(mode);
            dotAny.put(mode.toString(), modeDef.toAny());
          }
        }
      }
      return dotAny;
    } catch (final Exception ex) {
      throw new IllegalStateException("Failed to create Any object for data object type " + _name, ex);
    }
  }

  /**
   * @param mode
   *          the mode
   * @return the object expression for the given mode
   */
  public ValueExpression getObject(final Mode mode) {
    if (_modeDefinitions.get(mode) != null) {
      return _modeDefinitions.get(mode).getObject();
    }
    return null;
  }

  /**
   * @param mode
   *          the mode
   * @return the store expression for the given mode
   */
  public ValueExpression getStore(final Mode mode) {
    if (_modeDefinitions.get(mode) != null) {
      return _modeDefinitions.get(mode).getStore();
    }
    return null;
  }

  /**
   * @return true, if data object type has persistent mode definition.
   */
  public boolean hasPersistentMode() {
    return _modeDefinitions.get(Mode.PERSISTENT) != null;
  }

  /**
   * @return true, if data object type has transient mode definition.
   */
  public boolean hasTransientMode() {
    return _modeDefinitions.get(Mode.TRANSIENT) != null;
  }

}
