/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.List;

import org.eclipse.smila.jobmanager.JobTaskProcessor;
import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.exceptions.NoSuchTaskGeneratorException;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.taskworker.PostponeTaskException;
import org.eclipse.smila.taskworker.RecoverableTaskException;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.Worker;

/**
 * Worker for asynchronous finishing of tasks.
 */
public class FinishTaskWorker implements Worker {

  /** worker name. */
  private static final String WORKER_NAME = TaskManager.FINISHING_TASKS_WORKER;

  /** the job manager to use for checking job state. */
  private JobTaskProcessor _jobTaskProcessor;

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return WORKER_NAME;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    try {
      final List<Task> newTasks = finishTask(taskContext);
      countNewTasks(taskContext, newTasks);
    } catch (final JobManagerException ex) {
      if (ex.isRecoverable()) {
        throw new RecoverableTaskException(ex.getMessage(), ex);
      } else {
        throw ex;
      }
    }
  }

  /** finish task and measure duration. */
  private List<Task> finishTask(final TaskContext taskContext) throws JobManagerException, PostponeTaskException {
    taskContext.addToCounter("finishedTaskCount", 1);
    final long start = taskContext.getTimestamp();
    try {
      return _jobTaskProcessor.finishTask(taskContext.getTask());
    } catch (final IllegalJobStateException e) {
      taskContext.getLog().warn(e.getMessage(), e);
      return null;
    } catch (final NoSuchTaskGeneratorException e) {
      // the respective bundle containing the task generator does not seem to have been activated, yet.
      // postpone the handling
      throw new PostponeTaskException("Could not find task generator. Maybe bundle has not yet been activated.", e);
    } catch (final JobManagerException e) {
      // check if the cause is the same as above and postpone it
      if (e.getCause() != null) {
        if (e.getCause() instanceof IllegalJobStateException) {
          taskContext.getLog().warn(e.getMessage(), e);
          return null;
        }
        if (e.getCause() instanceof NoSuchTaskGeneratorException) {
          throw new PostponeTaskException(
            "Could not find task generator. Maybe bundle has not yet been activated.", e);
        }
      }
      throw e;
    } finally {
      taskContext.measureTime("finishTask", start);
    }
  }

  /** set counter for new tasks. */
  private void countNewTasks(final TaskContext taskContext, final List<Task> newTasks) throws TaskmanagerException {
    if (newTasks != null && !newTasks.isEmpty()) {
      taskContext.addToCounter("newTaskCount", newTasks.size());
    }
  }

  /**
   * Sets the reference to the JobTaskProcessor service.
   * 
   * @param jobTaskProcessor
   *          the reference to the JobTaskProcessor service.
   */
  public void setJobTaskProcessor(final JobTaskProcessor jobTaskProcessor) {
    _jobTaskProcessor = jobTaskProcessor;
  }

  /**
   * Resets the reference to the JobTaskProcessor service to null if the given jobTaskProcessor instance is set.
   * 
   * @param jobTaskProcessor
   *          the reference to the current JobTaskProcessor service.
   */
  public void unsetJobTaskProcessor(final JobTaskProcessor jobTaskProcessor) {
    if (_jobTaskProcessor == jobTaskProcessor) {
      _jobTaskProcessor = null;
    }
  }
}
