/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.JobRun;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.JobTaskProcessor;
import org.eclipse.smila.jobmanager.definitions.Bucket;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Mode;
import org.eclipse.smila.jobmanager.definitions.WorkflowAction;
import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.exceptions.NoSuchTaskGeneratorException;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorageException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerationUtil;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerator;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorBase;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorProvider;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.ServiceUnavailableException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.taskworker.output.OutputMode;
import org.eclipse.smila.utils.collections.MultiValueMap;
import org.osgi.service.component.ComponentContext;

/**
 * Implements {@link JobTaskProcessor}.
 */
public class JobTaskProcessorImpl implements JobTaskProcessor {

  /** if this time is exceeded while generating tasks, a log message is written. */
  private static final long MIN_TASK_GENERATION_TIME_TO_LOG = 1000; // ms

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** service reference to definition persistence. */
  private DefinitionPersistence _defPersistence;

  /** service reference to definition storage. */
  private RunStorage _runStorage;

  /** service reference to taskmanager. */
  private TaskManager _taskManager;

  /** service reference to cluster config service to read job property. */
  private ClusterConfigService _clusterConfigService;

  /** service reference to object store. */
  private ObjectStoreService _objectStore;

  /** service reference to task generator provider used to select a task generator. */
  private TaskGeneratorProvider _taskGeneratorProvider;

  /** service reference to job run engine. */
  private JobRunEngine _runEngine;

  /**
   * OSGi Declarative Services service activation method.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void activate(final ComponentContext context) {
    final Collection<String> workers = _defPersistence.getWorkers();
    for (final String worker : workers) {
      try {
        _taskManager.addTaskQueue(worker);
      } catch (final TaskmanagerException ex) {
        _log.warn("Error creating task queue for worker '" + worker + "'", ex);
      }
    }
  }

  @Override
  public List<Task> finishTask(final Task currentTask) throws JobManagerException {
    final String workflowRunId = currentTask.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);
    final String jobRunId = currentTask.getProperties().get(Task.PROPERTY_JOB_RUN_ID);
    final String jobName = currentTask.getProperties().get(Task.PROPERTY_JOB_NAME);
    final String workerName = getOriginalWorkerName(currentTask);

    try {
      if (!_runStorage.hasTask(jobName, workflowRunId, currentTask.getTaskId())) {
        throw new IllegalJobStateException("Task '" + currentTask.getTaskId() + "' for job '" + jobName
          + "' and run '" + jobRunId + "' is unknown, maybe already finished or workflow run was canceled.");
      }
      try {
        return handleTask(currentTask, jobName, jobRunId, workflowRunId, workerName);
      } catch (final JobManagerException jme) {
        if (currentTask.getResultDescription().getStatus() != TaskCompletionStatus.FATAL_ERROR
          && !jme.isRecoverable()) {
          try {
            handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, false);
          } catch (final JobManagerException e) {
            _log.error("Exception while handling fatal error during nonrecoverable finishing exception.", e);
          }
        }
        throw jme;
      }
    } finally {
      try {
        if (checkAndHandleWorkflowRunCompleted(jobName, jobRunId, workflowRunId)) {
          _runEngine.checkAndHandleJobRunCompleted(jobName, jobRunId);
        }
      } catch (final JobManagerException e) {
        final String message = "Exception while checking workflow/job run completion.";
        _log.error(message, e);
        throw new JobManagerException(message, e, true); // recoverable to repeat the operation
      }
    }
  }

  @Override
  public Task getInitialTask(final String workerName, final String jobName) throws JobManagerException {
    String jobRunId = null;
    String workflowRunId = null;
    Task task = null;
    jobRunId = _runStorage.getJobRunId(jobName);
    if (jobRunId == null || !checkJobStateForTaskCreation(jobName, jobRunId, true)) {
      throw new IllegalJobStateException("Job with name '" + jobName + "' is not running or is already finishing.");
    }
    final JobRun jobRun = _runEngine.ensureJobRun(jobName, jobRunId);
    try {
      workflowRunId = _runEngine.startWorkflowRun(jobRun);
      task = getInitialTask(jobRun, workerName, workflowRunId);
      // initial tasks are not recoverable, because no other worker would ever retry them
      task.getProperties().put(Task.PROPERTY_RECOVERABLE, Boolean.toString(Boolean.FALSE));
      _runStorage.startTask(jobName, jobRunId, workflowRunId, workerName, task.getTaskId());
      storeTaskForBarriers(jobName, jobRunId, workflowRunId, task);
      _taskManager.addInProgressTask(task);
      return task;
    } catch (final Exception e) {
      cleanupFailedWorkflowRun(jobName, jobRunId, workflowRunId, workerName, task);
      throw new JobManagerException("Getting initial task failed for worker '" + workerName + "' for job '"
        + jobName + "' due to error.", e);
    }
  }

  /** @return value of task property {@link Task#PROPERTY_ORIGINAL_WORKER}. */
  private String getOriginalWorkerName(final Task currentTask) {
    return currentTask.getProperties().get(Task.PROPERTY_ORIGINAL_WORKER);
  }

  private List<Task> handleTask(final Task currentTask, final String jobName, final String jobRunId,
    final String workflowRunId, final String workerName) throws JobManagerException {
    final ResultDescription resultDescription = currentTask.getResultDescription();
    switch (resultDescription.getStatus()) {
      case SUCCESSFUL:
        if (_log.isTraceEnabled()) {
          _log.trace("Successfully handles task '" + currentTask.getTaskId() + "' of worker '" + workerName
            + "' in job run '" + jobRunId + "' of job '" + jobName + "'");
        }
        return handleSuccessfulTask(jobName, jobRunId, workflowRunId, workerName, currentTask,
          resultDescription.getCounters());
      case OBSOLETE:
        if (_log.isTraceEnabled()) {
          _log.trace("Obsolete task '" + currentTask.getTaskId() + "' of worker '" + workerName + "' in job run '"
            + jobRunId + "' of job '" + jobName + "'");
        }
        return handleObsoleteTask(jobName, jobRunId, workflowRunId, workerName, currentTask);
      case RECOVERABLE_ERROR:
        // recoverable error - try to repeat the task processing
        _log.warn("A recoverable error '" + resultDescription.getErrorCode() + "'('"
          + resultDescription.getErrorMessage() + "') occurred in processing of task '" + currentTask.getTaskId()
          + "' for worker '" + workerName + "'");
        return handleRecoverableTaskError(jobName, jobRunId, workflowRunId, workerName, currentTask,
          resultDescription);
      default:
      case FATAL_ERROR:
        // fatal error - don't try to repeat the task processing
        _log.error("A fatal error '" + resultDescription.getErrorCode() + "'('"
          + resultDescription.getErrorMessage() + "') occurred in processing of task " + currentTask.getTaskId()
          + " of worker " + workerName + ". Workflow run '" + workflowRunId
          + "' will be marked as failed, its tasks will be canceled.");
        handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, false);
        return new ArrayList<Task>();
    }
  }

  /**
   * @return a step id which is unique for a job run.
   */
  private String getStepId(final String workerName, final Task currentTask) {
    return currentTask.getProperties().get(Task.PROPERTY_ACTION_POS) + "_" + workerName;
  }

  /**
   * @return the follow up tasks of the task that is successfully finished.
   */
  private List<Task> handleSuccessfulTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String workerName, final Task currentTask, final Map<String, Number> workerCounter)
    throws JobManagerException {

    final List<Task> followUpTasks = new ArrayList<Task>();
    try {
      final JobRun jobRun = _runEngine.ensureJobRun(jobName, jobRunId);
      final WorkerDefinition worker = _defPersistence.getWorker(workerName);

      if (!checkJobStateForTaskFinishing(jobName, jobRunId,
        currentTask.getProperties().containsKey(Task.PROPERTY_IS_COMPLETING_TASK))) {
        throw new IllegalJobStateException("Could not finish Task. Job with name'" + jobName
          + "' is not running or is already completed.");
      }
      // create follow up tasks if we may do so:
      if (checkJobStateForTaskCreation(jobName, jobRunId, false)) {
        adaptBulksForMultipleOutputSlots(currentTask, workerName);
        followUpTasks.addAll(createFollowUpTasks(currentTask, jobRun, workflowRunId));
        final List<Task> uniqueFollowUpTasks = _taskManager.filterDuplicates(followUpTasks);
        storeTasks(uniqueFollowUpTasks);
      }
      notifyTaskGenerator(worker, currentTask, TaskCompletionStatus.SUCCESSFUL);
      followUpTasks.addAll(checkBarriers(jobName, jobRunId, workflowRunId, currentTask));
      // delete AFTER the new tasks have been added.
      // Otherwise some other concurrent invocation might think there are no tasks left and finishes the workflow run!
      _runStorage.finishTask(jobName, jobRunId, workflowRunId, getStepId(workerName, currentTask),
        currentTask.getTaskId(), workerCounter, currentTask.getProperties());
    } catch (final NoSuchTaskGeneratorException e) {
      // throw recoverable error, the bundle just might not be activated, yet.
      throw new JobManagerException("Could not find task generator while handling successful task '"
        + currentTask.getTaskId() + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e, true);
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling successful task '" + currentTask.getTaskId()
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
    return followUpTasks;
  }

  /**
   * finish obsolete task - doesn't create immediate followup tasks, but can still trigger a barrier
   * 
   * @return tasks for barrier actions that were triggered by finishing this task
   */
  private List<Task> handleObsoleteTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String workerName, final Task currentTask) throws JobManagerException {
    try {
      final WorkerDefinition worker = _defPersistence.getWorker(workerName);

      if (!checkJobStateForTaskFinishing(jobName, jobRunId,
        currentTask.getProperties().containsKey(Task.PROPERTY_IS_COMPLETING_TASK))) {
        throw new IllegalJobStateException("Could not finish obsolete task '" + currentTask.getTaskId()
          + "'. Job with name'" + jobName + "' is not running or is already completed.");
      }
      notifyTaskGenerator(worker, currentTask, TaskCompletionStatus.OBSOLETE);
      final List<Task> followUpTasks = checkBarriers(jobName, jobRunId, workflowRunId, currentTask);
      _runStorage.obsoleteTask(jobName, jobRunId, workflowRunId, getStepId(workerName, currentTask),
        currentTask.getTaskId(), currentTask.getProperties());
      return followUpTasks;
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling obsolete task '" + currentTask.getTaskId()
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
  }

  /**
   * @return follow up tasks. This could be the current task in case of a retry. If the number of retries exceeded no
   *         follow up tasks are generated. In case of mode AUTOCOMMIT task is handled as succeeded.
   */
  private List<Task> handleRecoverableTaskError(final String jobName, final String jobRunId,
    final String workflowRunId, final String workerName, final Task currentTask,
    final ResultDescription resultDescription) throws JobManagerException {

    final List<Task> followUpTasks = new ArrayList<Task>();
    try {
      final WorkerDefinition worker = _defPersistence.getWorker(workerName);
      if (worker == null) {
        throw new JobManagerException("Worker '" + workerName + "' not found while handling recoverable task '"
          + currentTask.getTaskId() + "' in job run '" + jobRunId + "' of job '" + jobName + "'.");
      }
      if (worker.getModes().contains(Mode.AUTOCOMMIT)) {
        // autoCommit=true -> commit task's output and handle task as "successful" processed
        handleRecoverableTaskWithAutocommit(jobName, jobRunId, workflowRunId, workerName, currentTask,
          resultDescription, followUpTasks);
      } else {
        // autoCommit=false -> repeat current task if it's recoverable
        if (currentTask.getProperties().containsKey(Task.PROPERTY_RECOVERABLE)
          && !Boolean.valueOf(currentTask.getProperties().get(Task.PROPERTY_RECOVERABLE))) {
          _log.warn("Could not retry failed task '" + currentTask.getTaskId() + "' cause it's not recoverable.");
          handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, false);
        } else {
          final int numberOfRetries =
            _runStorage.getTaskRetries(jobName, jobRunId, workflowRunId, getStepId(workerName, currentTask),
              currentTask.getTaskId()) + 1;
          // if we can retry it again, do so. If not, task is failed.
          if (numberOfRetries <= _clusterConfigService.getMaxRetries()) {
            final Task retryTask = currentTask.createRetryTask(TaskGeneratorBase.createTaskId());
            storeTask(retryTask, numberOfRetries);
            followUpTasks.add(retryTask);
            finishTaskForBarriers(_runEngine.ensureJobRun(jobName, jobRunId), workflowRunId, currentTask);
            _runStorage.retriedTask(jobName, jobRunId, workflowRunId, getStepId(workerName, currentTask),
              currentTask.getTaskId(),
              !TaskManager.TASKERROR_TIME_TO_LIVE.equals(resultDescription.getErrorCode()),
              currentTask.getProperties());
          } else {
            _log.error("Could not retry failed recoverable task '" + currentTask.getTaskId()
              + "' , because maximum number of retries was reached.");
            handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, true);
          }
        }
      }
    } catch (final TaskmanagerException | RuntimeException e) {
      throw new JobManagerException("Error while handling recoverable task '" + currentTask.getTaskId()
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
    return followUpTasks;
  }

  /**
   * recoverable error on an AUTOCOMMIT worker's task -> task is handled as succeeded.
   */
  private void handleRecoverableTaskWithAutocommit(final String jobName, final String jobRunId,
    final String workflowRunId, final String workerName, final Task currentTask,
    final ResultDescription resultDescription, final List<Task> followUpTasks) throws JobManagerException {
    if (_log.isInfoEnabled()) {
      _log.info("Task '" + currentTask.getTaskId() + "' of autocommit worker '" + workerName + "' in job '"
        + jobName + "' failed with an recoverable error: finishing it as successful.");
    }
    for (final Entry<String, List<BulkInfo>> entry : currentTask.getOutputBulks().entrySet()) {
      for (final BulkInfo bulkInfo : entry.getValue()) {
        final String storeName = bulkInfo.getStoreName();
        final String objectId = bulkInfo.getObjectName();
        // finish already created objects so the failing worker can no longer append data to it, if it's still alive.
        // and the data is closed cleanly.
        try {
          if (ObjectStoreRetryUtil.retryExistsObject(_objectStore, storeName, objectId)) {
            _objectStore.finishObject(storeName, objectId);
          }
        } catch (final ServiceUnavailableException ex) {
          _log.warn("Error finishing object '" + objectId + "' in store '" + storeName + "', retrying.", ex);
          throw new JobManagerException("Could not finish object '" + objectId + "' in store '" + storeName + "'.",
            ex);
        } catch (final Exception ex) {
          _log.warn("Error finishing object '" + objectId + "' in store '" + storeName
            + "', ignoring and continuing.", ex);
        }
      }
    }
    followUpTasks.addAll(handleSuccessfulTask(jobName, jobRunId, workflowRunId, workerName, currentTask,
      resultDescription.getCounters()));
  }

  /**
   * fatal error -> workflow run failed.
   */
  private void handleFatalError(final String jobName, final String jobRunId, final String workflowRunId,
    final String workerName, final Task currentTask, final boolean failedAfterRetry) throws JobManagerException {
    try {
      try {
        final WorkerDefinition worker = _defPersistence.getWorker(workerName);
        if (worker == null) {
          throw new JobManagerException("Worker '" + workerName + "' not found while handling fatal error task '"
            + currentTask.getTaskId() + "' in job run '" + jobRunId + "' of job '" + jobName + "'.");
        }
        notifyTaskGenerator(worker, currentTask, TaskCompletionStatus.FATAL_ERROR);
      } finally {
        // fail the workflow run - we have to make sure that only one thread to do this
        if (_log.isDebugEnabled()) {
          _log.debug("Workflow run '" + workflowRunId + "' failed, preparing finish");
        }
        final boolean doIt = _runStorage.prepareToFinishWorkflowRun(jobName, jobRunId, workflowRunId);
        try {
          // count and delete failed task
          // (after preparing wf run for finish, to avoid that someone else finishes the wf run after task is deleted!)
          _runStorage.failedTask(jobName, jobRunId, workflowRunId, getStepId(workerName, currentTask),
            currentTask.getTaskId(), failedAfterRetry, currentTask.getProperties());
        } catch (final Exception e) {
          if (doIt) {
            // we will continue to fail the workflow run
            _log.warn("Error while failing task '" + currentTask.getTaskId() + "' for workflow run '"
              + workflowRunId + "'");
          } else {
            throw e;
          }
        }
        if (doIt) {
          deleteFailedWorkflowRun(jobName, jobRunId, workflowRunId);
          _runEngine.checkAndHandleJobRunCompleted(jobName, jobRunId);
        }
        if (_log.isDebugEnabled()) {
          _log.debug("Workflow run '" + workflowRunId + "' failed");
        }
      }
      // TODO we have to do some smarter handling here:
      // - what to do with keep-alives from tasks of given workflow run?
      // - what to do with successful finished tasks of given workflow run?
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling failed task in workflow run '" + workflowRunId
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
  }

  private void deleteFailedWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    if (_log.isDebugEnabled()) {
      _log.debug("Workflow run '" + workflowRunId + "' failed, aggregating counters");
    }
    _runStorage.failedWorkflowRun(jobName, jobRunId, workflowRunId);
    if (_log.isDebugEnabled()) {
      _log.debug("Workflow run '" + workflowRunId + "' failed, deleting transient bulks");
    }
    _runEngine.deleteTransientBulks(jobName, jobRunId, workflowRunId);
    if (_log.isDebugEnabled()) {
      _log.debug("Workflow run '" + workflowRunId + "' failed, deleting data");
    }
    _runStorage.deleteWorkflowRunData(jobName, jobRunId, workflowRunId);
    if (_log.isDebugEnabled()) {
      _log.debug("Workflow run '" + workflowRunId + "' failed, deleting workflow run");
    }
    _runStorage.deleteWorkflowRun(jobName, jobRunId, workflowRunId);
    if (_log.isDebugEnabled()) {
      _log.debug("Workflow run '" + workflowRunId + "' failed, checking job run completion");
    }
  }

  /**
   * Checks if a workflow run is finished and handles finishing the workflow run. I.e. deleting transient buckets,
   * merging statistics.
   * 
   * This can not be done atomic with ZK, but the impl. is based on the idea that in case of a server crash (or sth.
   * similar) somewhere in between the whole operation should be repeatable. (And it will be repeated because the
   * finishing of the task that caused the method call will be repeated after a crash.). Operation is repeatable after a
   * crash cause we use an ephemeral zk node in prepareToFinishWorkflowRun().
   * 
   * The second idea is, that the finishing of a job run should not be refused if we crashed in between and couldn't
   * successfully remove all workflow run data from zk. Therefore the flags set in prepareToFinishWorkflowRun() are also
   * a marker that this workflow run is completed.
   * 
   * @return true if workflow run completed.
   */
  private boolean checkAndHandleWorkflowRunCompleted(final String jobName, final String jobRunId,
    final String workflowRunId) throws JobManagerException {
    final boolean completed = isWorkflowRunCompleted(jobName, jobRunId, workflowRunId);
    if (completed) {
      if (_log.isDebugEnabled()) {
        _log.debug("Workflow run '" + workflowRunId + "' completed, deleting transient bulks");
      }
      _runEngine.deleteTransientBulks(jobName, jobRunId, workflowRunId);
      if (_runStorage.hasWorkflowRun(jobName, jobRunId, workflowRunId)) {
        if (_log.isDebugEnabled()) {
          _log.debug("Workflow run '" + workflowRunId + "' completed, preparing finish");
        }
        final boolean doIt = _runStorage.prepareToFinishWorkflowRun(jobName, jobRunId, workflowRunId);
        // we have to make sure that only one thread does the finishing
        if (doIt) {
          if (_log.isDebugEnabled()) {
            _log.debug("Workflow run '" + workflowRunId + "' completed, aggregating counters");
          }
          _runStorage.successfulWorkflowRun(jobName, jobRunId, workflowRunId); // aggregate counters
          if (_log.isDebugEnabled()) {
            _log.debug("Workflow run '" + workflowRunId + "' completed, deleting data");
          }
          _runStorage.deleteWorkflowRunData(jobName, jobRunId, workflowRunId); // delete the data first
          if (_log.isDebugEnabled()) {
            _log.debug("Workflow run '" + workflowRunId + "' completed, deleting workflow run");
          }
          _runStorage.deleteWorkflowRun(jobName, jobRunId, workflowRunId); // delete workflow run, incl. finishing flag
        }
      }
      if (_log.isDebugEnabled()) {
        _log.debug("Workflow run '" + workflowRunId + "' completed");
      }
    }
    return completed;
  }

  /**
   * 
   * Checks if a task may be generated for the job run with the given jobRunId.
   * 
   * @param jobName
   *          The name of the job to check.
   * @param jobRunId
   *          The job id to check.
   * @param isInitialTask
   *          Is this task an initial task (true) or follow-up task(false).
   * @return true if tasks may be created, false if not.
   * @throws RunStorageException
   *           exception while accessing JobRun data.
   */
  private boolean checkJobStateForTaskCreation(final String jobName, final String jobRunId,
    final boolean isInitialTask) throws RunStorageException {
    final JobState jobState = _runStorage.getJobState(jobName, jobRunId);
    if (jobState == JobState.RUNNING) {
      return true;
    }
    if (!isInitialTask && jobState == JobState.FINISHING) {
      return true;
    }
    return false;
  }

  /**
   * 
   * Checks if a task may be generated for the job run with the given jobRunId.
   * 
   * @param jobName
   *          The name of the job to check.
   * @param jobRunId
   *          The job id to check.
   * @param isCompletingTasks
   *          Is this task a completing task (true) or standard task(false).
   * @return true if tasks may be created, false if not.
   * @throws RunStorageException
   *           exception while accessing JobRun data.
   */
  private boolean checkJobStateForTaskFinishing(final String jobName, final String jobRunId,
    final boolean isCompletingTasks) throws RunStorageException {
    final JobState jobState = _runStorage.getJobState(jobName, jobRunId);
    if (jobState == JobState.RUNNING || jobState == JobState.FINISHING || jobState == JobState.COMPLETING
      && isCompletingTasks) {
      return true;
    }
    return false;
  }

  /**
   * adapt current task's output bulks for multiple output slots for which the worker has created (more than one) output
   * buckets itself.
   */
  private void adaptBulksForMultipleOutputSlots(final Task currentTask, final String workerName)
    throws JobManagerException {
    final WorkerDefinition workerDef = _defPersistence.getWorker(workerName);
    for (final String slotName : currentTask.getOutputBulks().keySet()) {
      if (workerDef != null && workerDef.getOutputModes() != null
        && workerDef.getOutputModes().get(slotName) != null
        && workerDef.getOutputModes().get(slotName).contains(OutputMode.MULTIPLE)) {
        final List<BulkInfo> taskBulks = currentTask.getOutputBulks().get(slotName);
        final BulkInfo template = taskBulks.get(0);
        final String objectNamePrefix = template.getObjectName();
        try {
          final Collection<StoreObject> objects =
            ObjectStoreRetryUtil.retryGetStoreObjectInfos(_objectStore, template.getStoreName(), objectNamePrefix);
          taskBulks.clear();
          for (final StoreObject o : objects) {
            final BulkInfo info = new BulkInfo(template.getBucketName(), template.getStoreName(), o.getId());
            taskBulks.add(info);
          }
        } catch (final ObjectStoreException e) {
          throw new JobManagerException("Error while adapting output bulks for multiple output slots", e);
        }
      }
    }
  }

  /**
   * create follow up tasks.
   * 
   * @param currentTask
   *          the currently processed task
   * @param jobRun
   *          job run
   * @param workflowRunId
   *          The workflow run id
   * @return a list of follow up tasks
   * @throws JobManagerException
   *           error
   */
  private List<Task> createFollowUpTasks(final Task currentTask, final JobRun jobRun, final String workflowRunId)
    throws JobManagerException {

    // check if all non-optional output buckets have been created, and get all output bulks
    final MultiValueMap<String, BulkInfo> bucketNameToCreatedBulksMap =
      checkOutputBulksForCreation(jobRun, currentTask);

    // check for new tasks for another job
    final Collection<JobRun> triggeredJobs = new ArrayList<JobRun>();
    for (final Entry<String, List<BulkInfo>> bucketToBulksEntry : bucketNameToCreatedBulksMap.entrySet()) {
      final String bucketName = bucketToBulksEntry.getKey();
      if (jobRun.getBucket(bucketName).isPersistent()) {
        triggeredJobs.addAll(getJobsTriggeredByBucket(jobRun, bucketName));
      } else {
        storeTransientBulks(jobRun, workflowRunId, bucketToBulksEntry.getValue());
      }
    }
    final AnyMap taskParamsToCopy = getTaskParametersToCopy(currentTask);
    final List<Task> followUpTasks =
      getFollowupTasks(jobRun, workflowRunId, bucketNameToCreatedBulksMap, taskParamsToCopy);
    for (final JobRun triggeredJob : triggeredJobs) {
      try {
        final Collection<Task> triggeredTasks = getTriggeredInitialTasks(triggeredJob, bucketNameToCreatedBulksMap);
        if (!triggeredTasks.isEmpty()) {
          final String triggeredJobWorkflowRunId = _runEngine.startWorkflowRun(triggeredJob);
          for (final Task task : triggeredTasks) {
            task.getProperties().put(Task.PROPERTY_WORKFLOW_RUN_ID, triggeredJobWorkflowRunId);
          }
          followUpTasks.addAll(triggeredTasks);
        }
      } catch (final Exception e) {
        _log.error(
          "Tried to create tasks for potentially triggered jobs, but got an error. Job '"
            + triggeredJob.getJobName() + "' will not be triggered.", e);
      }
    }
    return followUpTasks;
  }

  /** invoke finishTask method of task generator that belongs to the given worker. */
  private void notifyTaskGenerator(final WorkerDefinition worker, final Task task, final TaskCompletionStatus status)
    throws TaskGeneratorException {
    final TaskGenerator taskGenerator = _taskGeneratorProvider.getTaskGenerator(worker);
    taskGenerator.finishTask(task, status);
  }

  /**
   * Checks if the workflow run is finished, i.e. no tasks for this workflow run are open.
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @param workflowRunId
   *          The id of the workflow run.
   * @return 'true' if workflow run is finished
   * @throws RunStorageException
   *           exception while accessing RunStorage.
   */
  private boolean isWorkflowRunCompleted(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    return !_runStorage.hasTasks(jobName, jobRunId, workflowRunId);
  }

  /**
   * Checks if non-'maybeEmpty' output bulks have been created. If a non-'maybeEmpty' output bulk has not been created
   * an exception is thrown. If a 'maybeEmpty' output bulk has not been created, it will be removed from the map.
   * 
   * A bulk has not been created, if it is empty or does not even exist. An empty optional bulk will be deleted. TODO
   * Stimmt das?
   * 
   * @param jobRun
   *          The JobRun instance for the current job run.
   * @param task
   *          The Task that should be checked.
   * @return output bulks without the not created optional buckets. (key: bucket name, value: BulkInfo)
   * @throws JobManagerException
   *           An error occurred while checking the objects or a non-optional bulk has not been created.
   */
  private MultiValueMap<String, BulkInfo> checkOutputBulksForCreation(final JobRun jobRun, final Task task)
    throws JobManagerException {
    final WorkerDefinition workerDef = _defPersistence.getWorker(getOriginalWorkerName(task));
    final MultiValueMap<String, BulkInfo> outputBulkMap = new MultiValueMap<String, BulkInfo>();
    final Iterator<Entry<String, List<BulkInfo>>> entryIter = task.getOutputBulks().entrySet().iterator();
    while (entryIter.hasNext()) {
      final Entry<String, List<BulkInfo>> currentEntry = entryIter.next();
      final String currentSlotName = currentEntry.getKey();
      final boolean maybeEmpty = workerDef.getOutput(currentSlotName).getModes().contains(OutputMode.MAYBEEMPTY);
      for (final BulkInfo bulkInfo : currentEntry.getValue()) {
        final String storeName = bulkInfo.getStoreName();
        final String objectName = bulkInfo.getObjectName();
        try {
          if (!ObjectStoreRetryUtil.retryExistsObject(_objectStore, storeName, objectName)) {
            if (!maybeEmpty) {
              throw new JobManagerException("Output bulk '" + objectName + "' in bucket '"
                + bulkInfo.getBucketName() + "' of worker '" + workerDef.getName() + "' (job '"
                + jobRun.getJobName() + "') has not been created but must not be empty.");
            }
          } else {
            // all's ok, add it to the output map.
            outputBulkMap.add(bulkInfo.getBucketName(), bulkInfo);
          }
        } catch (final Exception e) {
          throw new JobManagerException(e.getClass().getName() + " occurred while checking output buckets.", e);
        }
      }
    }
    return outputBulkMap;
  }

  /**
   * Returns a list of jobs (other than triggeringJob) where a startAction exists that is triggered by the persistent
   * bucket referenced in bucketStorageInfo.
   * 
   * @param triggeringJobRun
   *          the triggering job run
   * @param bucketName
   *          The BucketStorageInfo.
   * @return A list of job runs (other than triggeringJob) where a startAction exists that is triggered by the
   *         persistent bucket referenced in bucketStorageInfo.
   * @throws JobManagerException
   *           Error occurred while accessing JobDefinitions instances.
   */
  private Collection<JobRun> getJobsTriggeredByBucket(final JobRun triggeringJobRun, final String bucketName)
    throws JobManagerException {
    final Bucket triggeringBucket = triggeringJobRun.getBucket(bucketName);
    final Collection<String> jobNames = _runStorage.getTriggeredJobs(triggeringBucket.getBucketId());
    // ignore triggering job
    jobNames.remove(triggeringJobRun.getJobName());
    final Collection<JobRun> triggeredJobs = new ArrayList<JobRun>();
    for (final String jobName : jobNames) {
      try {
        final JobRunInfo runInfo = _runStorage.getJobRunInfo(jobName);
        if (runInfo != null && runInfo.getState() == JobState.RUNNING) {
          final String jobRunId = runInfo.getId();
          final JobRun jobRun = _runEngine.ensureJobRun(jobName, jobRunId);
          if (jobRun.isTriggeredBy(triggeringBucket)) {
            triggeredJobs.add(jobRun);
          }
        }
      } catch (final Exception e) {
        _log.warn("Tried to check potentially triggered jobs, but got an error. Job '" + jobName
          + "' will not be triggered.", e);
      }
    }
    return triggeredJobs;
  }

  /**
   * Stores transient bulks in RunStorage.
   * 
   * @param jobRun
   *          job run
   * @param workflowRunId
   *          the id of the workflow run.
   * @param transientBulks
   *          the bulks to store in the runStorage
   * @throws RunStorageException
   *           exception while storing the information in RunStorage
   */
  private void storeTransientBulks(final JobRun jobRun, final String workflowRunId,
    final List<BulkInfo> transientBulks) throws RunStorageException {
    final String jobName = jobRun.getJobName();
    final String jobRunId = jobRun.getJobRunId();
    for (final BulkInfo bulkInfo : transientBulks) {
      _runStorage.addTransientBulk(jobName, jobRunId, workflowRunId,
        bulkInfo.getStoreName() + "/" + bulkInfo.getObjectName());
    }
  }

  /** TODO remove this hack to copy task parameters in follow up tasks. */
  private AnyMap getTaskParametersToCopy(final Task currentTask) {
    final AnyMap taskParams = currentTask.getParameters();
    final AnyMap paramsToCopy = DataFactory.DEFAULT.createAnyMap();
    for (final Entry<String, Any> param : taskParams.entrySet()) {
      if (param.getKey().startsWith(JobManagerConstants.SYSTEM_PARAMETER_PREFIX)) {
        paramsToCopy.put(param.getKey(), param.getValue());
      }
    }
    return paramsToCopy;
  }

  /**
   * cleanup possibly created workflow run data after failure during start. Exceptions during clean up are logged.
   * 
   * @param jobName
   *          job name
   * @param jobRunId
   *          job run id
   * @param workflowRunId
   *          workflow run id. may be null, then nothing has to be done.
   * @param task
   *          inital task (maybe "null")
   */
  private void cleanupFailedWorkflowRun(final String jobName, final String jobRunId, final String workerName,
    final String workflowRunId, final Task task) {
    if (jobRunId != null && workflowRunId != null) {
      try {
        handleFatalError(jobName, jobRunId, workflowRunId, workerName, task, false);
      } catch (final Exception ex) {
        _log.error("Error during cleanup of failed new workflow run '" + jobRunId + "' for job '" + jobName + "'.",
          ex);
      }
    }
  }

  /**
   * Creates the initial task for this job run's new workflow run.
   * 
   * @param workerName
   *          The name of the worker requesting the initial task.
   * @param workflowRunId
   *          The id for the new workflow run.
   * @return An initial task for the new job run.
   * @throws JobManagerException
   *           An error occurred while checking if an initial task could be created or while creating the initial task.
   */
  private Task getInitialTask(final JobRun jobRun, final String workerName, final String workflowRunId)
    throws JobManagerException {
    final WorkflowAction startAction = jobRun.getStartAction();
    if (startAction == null || !workerName.equals(startAction.getWorker())
      || !_defPersistence.getWorker(workerName).getModes().contains(Mode.BULKSOURCE)) {
      throw new JobManagerException("Worker '" + workerName + "' is not defined as start action for job '"
        + jobRun.getJobName() + "' or has no '" + Mode.BULKSOURCE.toString() + "' mode.");
    }
    final List<Task> taskList =
      generateTasksForAction(jobRun, workflowRunId, new MultiValueMap<String, BulkInfo>(), startAction, null);
    // there must be only one task
    if (taskList.size() != 1) {
      throw new JobManagerException("Not exactly one initial task produced for worker '" + workerName
        + "' in workflow run '" + workflowRunId + "'.");
    }
    return taskList.get(0);
  }

  /**
   * Gets the follow-up tasks for workers within the actual workflow run depending on a list of bulks.
   * 
   * Note that all tasks returned by this method will have thir workflowRunId parameter set to "null" since there will
   * be no workflow run started. You will have to set this parameter for any task in the returned collection.
   * 
   * @param bucketNameToReallyCreatedBulksMap
   *          The bulks potentially triggering tasks for follow up workers.
   * @return A collection of tasks triggered by the bulks within this workflow for the given workflow run.
   */
  private List<Task> getTriggeredInitialTasks(final JobRun jobRun,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap) throws JobManagerException {
    // only for the initial action
    // note: we are setting null as workflow run, the workflow-run-id has to be set outside!
    return generateTasksForAction(jobRun, null, bucketNameToReallyCreatedBulksMap, jobRun.getStartAction(), null);
  }

  /**
   * Gets the follow-up tasks for workers within the actual workflow run depending on a list of bulks.
   * 
   * @param workflowRunId
   *          The id of the current workflow run.
   * @param bucketNameToReallyCreatedBulksMap
   *          The bulks potentially triggering tasks for follow up workers.
   * @param taskParamsToCopy
   *          the parameters from the current task to copy to the follow up tasks
   * @return A collection of tasks triggered by the bulks within this workflow for the given workflow run. error while
   *         generating new tasks
   */
  private List<Task> getFollowupTasks(final JobRun jobRun, final String workflowRunId,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap, final AnyMap taskParamsToCopy)
    throws TaskGeneratorException {
    final List<Task> followUpTasks = new ArrayList<Task>();
    // check for follow up workflow actions for given bulks
    final Collection<WorkflowAction> followUpActions =
      getFollowupWorkflowActions(jobRun, bucketNameToReallyCreatedBulksMap.keySet());

    // for all non-barrier follow up actions: find appropriate TaskGenerator and generate follow up tasks
    for (final WorkflowAction action : followUpActions) {
      if (!jobRun.getBarrierActions().contains(action)) {
        followUpTasks.addAll(generateTasksForAction(jobRun, workflowRunId, bucketNameToReallyCreatedBulksMap,
          action, taskParamsToCopy));
      }
    }
    return followUpTasks;
  }

  /**
   * Returns a set of follow up actions within a workflow. Follow up actions have to be unique!
   * 
   * @param bucketNames
   *          a collection of bucket names,.
   * @return A set of unique follow-up actions. Empty (but not null) if there are none.
   */
  private Collection<WorkflowAction> getFollowupWorkflowActions(final JobRun jobRun,
    final Collection<String> bucketNames) {
    final Set<WorkflowAction> followUpActions = new HashSet<WorkflowAction>();

    for (final String bucketName : bucketNames) {
      final Collection<WorkflowAction> actions = jobRun.getTriggeredActionsForBucket(bucketName);
      if (actions != null) {
        followUpActions.addAll(actions);
      }
    }
    return followUpActions;
  }

  /**
   * Generates a task for an action.
   * 
   * @param workflowRunId
   *          the workflow run id
   * @param bucketNameToReallyCreatedBulksMap
   *          the map from bucket name as key to BulkInfo-List as value
   * @param action
   *          the action to generate the tasks for
   * @param taskParamsToCopy
   *          the parameters from the current task to copy to the follow up tasks
   * @return A list of tasks for an action
   * @throws TaskGeneratorException
   *           error while generating new tasks
   */
  private List<Task> generateTasksForAction(final JobRun jobRun, final String workflowRunId,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap, final WorkflowAction action,
    final AnyMap taskParamsToCopy) throws TaskGeneratorException {

    final WorkerDefinition workerDef = _defPersistence.getWorker(action.getWorker());
    final TaskGenerator taskGenerator = _taskGeneratorProvider.getTaskGenerator(workerDef);

    // get parameters
    final AnyMap paramExpressions = DataFactory.DEFAULT.createAnyMap();
    if (jobRun.getParameters() != null) {
      paramExpressions.putAll(jobRun.getParameters());
    }
    if (action.getParameters() != null) {
      paramExpressions.putAll(action.getParameters());
    }
    final AnyMap parameters = ExpressionUtil.evaluateParameters(paramExpressions);
    if (taskParamsToCopy != null) {
      parameters.putAll(taskParamsToCopy);
    }

    final Map<String, Bucket> inputSlotNameToBucket = jobRun.getInputBucketsForAction(action);
    final Map<String, Bucket> outputSlotNameToBucketMap = jobRun.getOutputBucketsForAction(action);

    final MultiValueMap<String, BulkInfo> changedInputSlotToBulkListMap =
      prepareInputSlots(action, bucketNameToReallyCreatedBulksMap);

    // let the task generator create the tasks
    final long start = System.currentTimeMillis();
    final List<Task> tasks =
      taskGenerator.createTasks(changedInputSlotToBulkListMap, inputSlotNameToBucket, outputSlotNameToBucketMap,
        parameters, action.getWorker());
    final long t = System.currentTimeMillis() - start;
    if (t > MIN_TASK_GENERATION_TIME_TO_LOG && _log.isInfoEnabled()) {
      _log.info("TaskGenerator '" + taskGenerator.getName() + "' time for creating tasks: " + t);
    }
    TaskGenerationUtil.setAdditionalTaskProperties(tasks, jobRun, workflowRunId, workerDef, action.getPosition());

    return tasks;
  }

  /**
   * associate the new input bulks that triggered this action with the slotnames.
   * 
   * @param action
   *          the action to generate the tasks for
   * @param bucketNameToReallyCreatedBulksMap
   *          the map from bucket name as key to BulkInfo-List as value
   * @return changedInputSlotToBulkListMap
   */
  private MultiValueMap<String, BulkInfo> prepareInputSlots(final WorkflowAction action,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap) {
    // prepare changed input buckets for new task
    final MultiValueMap<String, BulkInfo> changedInputSlotToBulkListMap = new MultiValueMap<String, BulkInfo>();
    final Map<String, String> actionInput = action.getInput();
    if (actionInput != null) {
      for (final Map.Entry<String, String> actionInputEntry : actionInput.entrySet()) {
        final List<BulkInfo> bulkInfoList = bucketNameToReallyCreatedBulksMap.get(actionInputEntry.getValue());
        if (bulkInfoList != null) {
          for (final BulkInfo inputBulk : bulkInfoList) {
            if (inputBulk != null) {
              if (!changedInputSlotToBulkListMap.containsKey(actionInputEntry.getKey())) {
                changedInputSlotToBulkListMap.put(actionInputEntry.getKey(), new ArrayList<BulkInfo>());
              }
              changedInputSlotToBulkListMap.get(actionInputEntry.getKey()).add(inputBulk);
            }
          }
        }
      }
    }
    return changedInputSlotToBulkListMap;
  }

  private void storeTasks(final List<Task> newTasks) throws JobManagerException, TaskmanagerException {
    for (final Task newTask : newTasks) {
      storeTask(newTask, 0);
    }
  }

  private void storeTask(final Task newTask, final int numberOfRetries) throws JobManagerException,
    TaskmanagerException {
    final String newTaskJobName = newTask.getProperties().get(Task.PROPERTY_JOB_NAME);
    final String newTaskJobRunId = newTask.getProperties().get(Task.PROPERTY_JOB_RUN_ID);
    final String newTaskWfRunId = newTask.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);

    // TODO find a clean solution for handling the following statements in a more "atomic" way.
    // If RunStorage task entry creation is done first (as is), failures can happen if process/node is crashing
    // right between the two statements: Task entry in RunStorage will exist there until job run is canceled.
    // If instead Taskmanager task creation would be done first, this may cause bug #3566.
    _runStorage.startTask(newTaskJobName, newTaskJobRunId, newTaskWfRunId, newTask.getWorkerName(),
      newTask.getTaskId(), numberOfRetries);
    storeTaskForBarriers(newTaskJobName, newTaskJobRunId, newTaskWfRunId, newTask);
    _taskManager.addTask(newTask);
    // TODO is this order of operations "OK"?. Changed order because of problems in cluster
  }

  private void storeTaskForBarriers(final String jobName, final String jobRunId, final String workflowRunId,
    final Task task) throws JobManagerException {
    if (!task.getProperties().containsKey(Task.PROPERTY_IS_COMPLETING_TASK)) {
      final JobRun jobRun = _runEngine.ensureJobRun(jobName, jobRunId);
      if (jobRun.hasBarriers()) {
        final int position = Integer.parseInt(task.getProperties().get(Task.PROPERTY_ACTION_POS));
        final WorkflowAction action = jobRun.getAction(position);
        final Collection<WorkflowAction> barriers = jobRun.getBarriersForAction(action);
        if (barriers != null) {
          for (final WorkflowAction barrier : barriers) {
            _runStorage.addTaskForBarrier(jobName, workflowRunId, barrier.getPosition(), task.getTaskId());
          }
        }
      }
    }
  }

  private List<Task> checkBarriers(final String jobName, final String jobRunId, final String workflowRunId,
    final Task currentTask) throws JobManagerException {
    if (!currentTask.getProperties().containsKey(Task.PROPERTY_IS_COMPLETING_TASK)) {
      final JobRun jobRun = _runEngine.ensureJobRun(jobName, jobRunId);
      if (jobRun.hasBarriers()) {
        final Collection<WorkflowAction> affectedBarriers =
          finishTaskForBarriers(jobRun, workflowRunId, currentTask);
        if (affectedBarriers != null && checkJobStateForTaskCreation(jobName, jobRunId, false)) {
          final List<Task> barrierTasks = new ArrayList<>();
          final List<WorkflowAction> uncheckedBarriers = new ArrayList<>(affectedBarriers);
          for (final WorkflowAction barrierAction : affectedBarriers) {
            // iterate on copy because list is modified in checkBarrier
            checkBarrier(jobRun, workflowRunId, barrierAction, uncheckedBarriers, barrierTasks);
          }
          return barrierTasks;
        }
      }
    }
    return Collections.emptyList();

  }

  /** store output bulks and remove task from barriers. */
  private Collection<WorkflowAction> finishTaskForBarriers(final JobRun jobRun, final String workflowRunId,
    final Task task) throws JobManagerException {
    final int position = Integer.parseInt(task.getProperties().get(Task.PROPERTY_ACTION_POS));
    final WorkflowAction action = jobRun.getAction(position);
    if (task.getResultDescription().getStatus() == TaskCompletionStatus.SUCCESSFUL) {
      storeOutputBulksForBarriers(jobRun, workflowRunId, task, action);
    }
    return removeTaskFromBarriers(jobRun, workflowRunId, task, action);
  }

  private void storeOutputBulksForBarriers(final JobRun jobRun, final String workflowRunId, final Task task,
    final WorkflowAction action) throws RunStorageException {
    for (final Map.Entry<String, List<BulkInfo>> outputBulks : task.getOutputBulks().entrySet()) {
      final String slotName = outputBulks.getKey();
      final String bucketName =
        jobRun.getOutputBucketsForAction(action).get(slotName).getBucketDefinition().getName();
      final Collection<WorkflowAction> triggeredActions = jobRun.getTriggeredActionsForBucket(bucketName);
      if (triggeredActions != null) {
        for (final WorkflowAction triggeredAction : triggeredActions) {
          if (jobRun.getBarrierActions().contains(triggeredAction)) {
            for (final BulkInfo outputBulk : outputBulks.getValue()) {
              _runStorage.addBulkForBarrier(jobRun.getJobName(), workflowRunId, triggeredAction.getPosition(),
                bucketName + "::" + outputBulk.getObjectName());
            }
          }
        }
      }
    }
  }

  private Collection<WorkflowAction> removeTaskFromBarriers(final JobRun jobRun, final String workflowRunId,
    final Task task, final WorkflowAction action) throws RunStorageException {
    final Collection<WorkflowAction> barriers = jobRun.getBarriersForAction(action);
    if (barriers != null) {
      for (final WorkflowAction barrier : barriers) {
        _runStorage.removeTaskForBarrier(jobRun.getJobName(), workflowRunId, barrier.getPosition(),
          task.getTaskId());
      }
    }
    return barriers;
  }

  private void checkBarrier(final JobRun jobRun, final String workflowRunId, final WorkflowAction barrierAction,
    final Collection<WorkflowAction> uncheckedBarriers, final List<Task> barrierTasks) throws JobManagerException {
    if (uncheckedBarriers.remove(barrierAction)) {
      checkPrecedingBarriers(jobRun, workflowRunId, barrierAction, uncheckedBarriers, barrierTasks);
      final String jobName = jobRun.getJobName();
      final int barrierPosition = barrierAction.getPosition();
      if (!_runStorage.tasksLockBarrier(jobName, workflowRunId, barrierPosition)
        && _runStorage.prepareOpeningOfBarrier(jobName, workflowRunId, barrierPosition)) {
        final List<Task> actionTasks = generateTasksForBarrierAction(jobRun, workflowRunId, barrierAction);
        try {
          storeTasks(actionTasks);
        } catch (final TaskmanagerException e) {
          throw new JobManagerException("Error while storing tasks for barrier '" + barrierAction.getWorker() + "@"
            + barrierPosition + "' in job run '" + jobRun.getJobRunId() + "' of job '" + jobName + "'.", e);
        }
        _runStorage.cleanOpenedBarrier(jobRun.getJobName(), workflowRunId, barrierPosition);
        barrierTasks.addAll(actionTasks);
      }
    }
  }

  private void checkPrecedingBarriers(final JobRun jobRun, final String workflowRunId,
    final WorkflowAction barrierAction, final Collection<WorkflowAction> uncheckedBarriers,
    final List<Task> barrierTasks) throws JobManagerException {
    final Collection<WorkflowAction> precedingBarriers = jobRun.getPrecedingBarriers(barrierAction);
    if (precedingBarriers != null) {
      for (final WorkflowAction precedingBarrier : precedingBarriers) {
        checkBarrier(jobRun, workflowRunId, precedingBarrier, uncheckedBarriers, barrierTasks);
      }
    }
  }

  private List<Task> generateTasksForBarrierAction(final JobRun jobRun, final String workflowRunId,
    final WorkflowAction action) throws TaskGeneratorException {
    try {
      final Collection<String> bulkIds =
        _runStorage.getBulksForBarrier(jobRun.getJobName(), workflowRunId, action.getPosition());
      if (!bulkIds.isEmpty()) {
        final MultiValueMap<String, BulkInfo> barrierBulks = new MultiValueMap<>();
        for (final String bucketBulkId : bulkIds) {
          final int index = bucketBulkId.indexOf("::");
          final String bucketName = bucketBulkId.substring(0, index);
          final String objectName = bucketBulkId.substring(index + 2);
          final String storeName = jobRun.getBucket(bucketName).getStoreName();
          final BulkInfo barrierBulk = new BulkInfo(bucketName, storeName, objectName);
          barrierBulks.add(bucketName, barrierBulk);
        }
        final List<Task> newTasks = generateTasksForAction(jobRun, workflowRunId, barrierBulks, action, null);
        return newTasks;
      } else {
        return Collections.emptyList();
      }
    } catch (final RunStorageException ex) {
      throw new TaskGeneratorException("Could not get bulks for barrier task creation", ex);
    }
  }

  /** set OSGI service. */
  public void setDefinitionPersistence(final DefinitionPersistence defPersistence) {
    _defPersistence = defPersistence;
  }

  /** unset OSGI service. */
  public void unsetDefinitionPersistence(final DefinitionPersistence defPersistence) {
    if (_defPersistence == defPersistence) {
      _defPersistence = null;
    }
  }

  /** set OSGI service. */
  public void setRunStorage(final RunStorage runStorage) {
    _runStorage = runStorage;
  }

  /** unset OSGI service. */
  public void unsetRunStorage(final RunStorage runStorage) {
    if (_runStorage == runStorage) {
      _runStorage = null;
    }
  }

  /** set OSGI service. */
  public void setObjectStoreService(final ObjectStoreService objectStore) {
    _objectStore = objectStore;
  }

  /** unset OSGI service. */
  public void unsetObjectStoreService(final ObjectStoreService objectStore) {
    if (_objectStore == objectStore) {
      _objectStore = null;
    }
  }

  /** set OSGI service. */
  public void setTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    _taskGeneratorProvider = taskGeneratorProvider;
  }

  /** unset OSGI service. */
  public void unsetTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    if (_taskGeneratorProvider == taskGeneratorProvider) {
      _taskGeneratorProvider = null;
    }
  }

  /** set OSGI service. */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /** unset OSGI service. */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }

  /** set OSGI service. */
  public void setClusterConfigService(final ClusterConfigService clusterConfigService) {
    _clusterConfigService = clusterConfigService;
  }

  /** unset OSGI service. */
  public void unsetClusterConfigService(final ClusterConfigService clusterConfigService) {
    if (_clusterConfigService == clusterConfigService) {
      _clusterConfigService = null;
    }
  }

  /** set OSGI service. */
  public void setJobRunEngine(final JobRunEngine runEngine) {
    _runEngine = runEngine;
  }

  /** unset OSGI service. */
  public void unsetJobRunEngine(final JobRunEngine runEngine) {
    if (_runEngine == runEngine) {
      _runEngine = null;
    }
  }

}
