/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Drazen Cindric (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.processing.bpel.test;

import java.io.IOException;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.DeleteMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.http.server.util.RequestHandler;
import org.eclipse.smila.processing.httphandler.PipelineHandler;
import org.eclipse.smila.test.DeclarativeServiceTestCase;

/**
 * Implements pipeline definition handler tests to see if workflow definitions are returned correctly.
 * 
 * @author drazen
 * 
 */
public class TestPipelineDefinitionHandler extends DeclarativeServiceTestCase {

  /** url for testing. */
  private static final String END_POINT = "http://localhost:8080/smila/pipeline/";

  /** JSON -> Any reader. */
  private final IpcAnyReader _anyReader = new IpcAnyReader();

  /** Any -> JSON serializer. */
  private final IpcAnyWriter _anyWriter = new IpcAnyWriter();

  private final HttpClient _httpclient = new HttpClient();

  /**
   * Test if service was successfully started and registered.
   * 
   * @throws Exception
   *           no service found.
   */
  public void testService() throws Exception {
    final RequestHandler service = getService(RequestHandler.class, "(uriPattern=/pipeline/\\([^/]+\\)/?$)");
    assertNotNull(service);
    assertTrue(service instanceof PipelineHandler);
  }

  /** test list of pipelines received from handler. */
  public void testPipelineList() throws Exception {
    final String uri = END_POINT;
    final HttpClient httpclient = new HttpClient();
    final GetMethod getMethod = new GetMethod(uri);
    try {
      httpclient.executeMethod(getMethod);
      assertEquals(HttpStatus.SC_OK, getMethod.getStatusCode());
      final AnyMap resultMap = readResultMap(getMethod);
      assertTrue(resultMap.containsKey("pipelines"));
      final AnySeq pipelines = resultMap.getSeq("pipelines");
      assertTrue(pipelines.size() > 8);
      for (final Any pipeline : pipelines) {
        assertTrue(pipeline.isMap());
        final AnyMap pipelineMap = (AnyMap) pipeline;
        final String name = pipelineMap.getStringValue("name");
        assertNotNull(name);
        final String url = pipelineMap.getStringValue("url");
        assertNotNull(url);
        assertTrue(url.startsWith(END_POINT));
        assertTrue(url.endsWith(name + "/"));
      }
    } finally {
      getMethod.releaseConnection();
    }

  }

  /**
   * Tests if definition of HelloWorldPipeline will be returned correctly.
   * 
   * @throws Exception
   *           an exception if something went wrong
   */
  public void testGetWithHelloWorldPipeline() throws Exception {
    final String uri = END_POINT + TestHelloWorldPipeline.PIPELINE_NAME;
    getPipeline(uri, TestHelloWorldPipeline.PIPELINE_NAME);
  }

  /**
   * Tests if expected status code and message will be returned for not existing pipeline name.
   * 
   * @throws Exception
   *           an exception if something went wrong
   */
  public void testNotExistingPipelineName() throws Exception {
    final String uri = END_POINT + "NotExistingPipe";
    checkPipelineUndefined(uri);
  }

  /**
   * test definition of a new pipeline via ReST API.
   */
  public void testDefinePipeline() throws Exception {
    final String pipelineName = "testDefinePipeline";
    final String pipelineBpel = AWorkflowProcessorTest.copyEchoPipeline(pipelineName);
    final AnyMap postResult = definePipeline(pipelineName, pipelineBpel);
    final AnyMap getResult = getPipeline(postResult.getStringValue("url"), pipelineName, pipelineBpel);
    assertEquals(postResult.getStringValue("timestamp"), getResult.getStringValue("timestamp"));
  }

  /**
   * test definition of a new pipeline without request body via ReST API.
   */
  public void testDefinePipelineEmpty() throws Exception {
    final PostMethod postMethod = new PostMethod(END_POINT);
    postMethod.setRequestEntity(new StringRequestEntity(_anyWriter.writeJsonObject(DataFactory.DEFAULT
      .createAnyMap()), "application/json", null));
    try {
      _httpclient.executeMethod(postMethod);
      assertEquals(HttpStatus.SC_BAD_REQUEST, postMethod.getStatusCode());
    } finally {
      postMethod.releaseConnection();
    }
  }

  /**
   * test definition of a new pipeline via ReST API.
   */
  public void testDeletePipeline() throws Exception {
    final String pipelineName = "testDeletePipeline";
    final String pipelineBpel = AWorkflowProcessorTest.copyEchoPipeline(pipelineName);
    final AnyMap postResult = definePipeline(pipelineName, pipelineBpel);
    getPipeline(postResult.getStringValue("url"), pipelineName, pipelineBpel);
    final DeleteMethod deleteRequest = new DeleteMethod(postResult.getStringValue("url"));
    try {
      _httpclient.executeMethod(deleteRequest);
      assertEquals(HttpStatus.SC_OK, deleteRequest.getStatusCode());
    } finally {
      deleteRequest.releaseConnection();
    }
    checkPipelineUndefined(postResult.getStringValue("url"));
  }

  /**
   * test definition of a new pipeline via ReST API.
   */
  public void testDeleteUndefinedPipeline() throws Exception {
    final String pipelineName = "testDeleteUndefinedPipeline";
    final String url = END_POINT + pipelineName;
    checkPipelineUndefined(url);
    final DeleteMethod deleteRequest = new DeleteMethod(url);
    try {
      _httpclient.executeMethod(deleteRequest);
      assertEquals(HttpStatus.SC_OK, deleteRequest.getStatusCode());
    } finally {
      deleteRequest.releaseConnection();
    }
    checkPipelineUndefined(url);
  }

  /**
   * test update of pipeline.
   */
  public void testUpdatePipeline() throws Exception {
    final String pipelineName = "testUpdatePipeline";
    final String pipelineBpel = AWorkflowProcessorTest.copyEchoPipeline(pipelineName);
    final AnyMap defineResult = definePipeline(pipelineName, pipelineBpel);
    getPipeline(defineResult.getStringValue("url"), pipelineName, pipelineBpel);
    Thread.sleep(100); // wait so that timestamp will increase.
    final AnyMap updateResult = definePipeline(pipelineName, pipelineBpel);
    getPipeline(defineResult.getStringValue("url"), pipelineName, pipelineBpel);
    assertTrue(defineResult.getDateTimeValue("timestamp").compareTo(updateResult.getDateTimeValue("timestamp")) < 0);
  }

  /** test error on invalid BPEL syntax. */
  public void testInvalidSyntax() throws Exception {
    final String pipelineName = "testInvalidSyntax";
    final AnyMap definition = DataFactory.DEFAULT.createAnyMap();
    definition.put("name", pipelineName);
    definition.put("definition", "This is no Bi Pel.");
    final PostMethod postMethod = new PostMethod(END_POINT);
    postMethod.setRequestEntity(new StringRequestEntity(_anyWriter.writeJsonObject(definition), "application/json",
      null));
    try {
      _httpclient.executeMethod(postMethod);
      assertEquals(HttpStatus.SC_BAD_REQUEST, postMethod.getStatusCode());
    } finally {
      postMethod.releaseConnection();
    }
  }

  /** test error on update for predefined pipeline. */
  public void testUpdatePredefinedPipeline() throws Exception {
    final String pipelineName = TestHelloWorldPipeline.PIPELINE_NAME;
    final AnyMap definition =
      AWorkflowProcessorTest.createWorkflowDefinition(pipelineName,
        AWorkflowProcessorTest.copyEchoPipeline(pipelineName));
    final PostMethod postMethod = new PostMethod(END_POINT);
    postMethod.setRequestEntity(new StringRequestEntity(_anyWriter.writeJsonObject(definition), "application/json",
      null));
    try {
      _httpclient.executeMethod(postMethod);
      assertEquals(HttpStatus.SC_BAD_REQUEST, postMethod.getStatusCode());
    } finally {
      postMethod.releaseConnection();
    }
  }

  /**
   * define a new pipeline and assert expected result.
   */
  private AnyMap definePipeline(final String pipelineName, final String pipelineBpel) throws IOException {
    final AnyMap definition = AWorkflowProcessorTest.createWorkflowDefinition(pipelineName, pipelineBpel);
    final PostMethod postMethod = new PostMethod(END_POINT);
    postMethod.setRequestEntity(new StringRequestEntity(_anyWriter.writeJsonObject(definition), "application/json",
      null));
    final String pipelineUrl;
    try {
      _httpclient.executeMethod(postMethod);
      assertEquals(HttpStatus.SC_CREATED, postMethod.getStatusCode());
      final AnyMap result = readResultMap(postMethod);
      assertEquals(pipelineName, result.getStringValue("name"));
      assertNotNull(result.getDateTimeValue("timestamp"));
      pipelineUrl = result.getStringValue("url");
      assertEquals(END_POINT + pipelineName + "/", pipelineUrl);
      return result;
    } finally {
      postMethod.releaseConnection();
    }
  }

  /**
   * get pipeline definition and compare to expected values.
   */
  private AnyMap getPipeline(final String pipelineUrl, final String expectedPipelineName,
    final String expectedPipelineBpel) throws IOException {
    final AnyMap result = getPipeline(pipelineUrl, expectedPipelineName);
    AWorkflowProcessorTest.assertWorkflowDefinition(expectedPipelineName, expectedPipelineBpel, result);
    return result;
  }

  /**
   * get pipeline definition and check name structure.
   */
  private AnyMap getPipeline(final String pipelineUrl, final String expectedPipelineName) throws IOException {
    AnyMap result;
    final GetMethod getMethod = new GetMethod(pipelineUrl);
    try {
      _httpclient.executeMethod(getMethod);
      assertEquals(HttpStatus.SC_OK, getMethod.getStatusCode());
      result = readResultMap(getMethod);
      assertEquals(expectedPipelineName, result.getStringValue("name"));
      assertNotNull(result.getStringValue("definition"));
      return result;
    } finally {
      getMethod.releaseConnection();
    }
  }

  /** check NOT_FOUND response for undefined pipelines. */
  private void checkPipelineUndefined(final String uri) throws IOException {
    final GetMethod getMethod = new GetMethod(uri);
    try {
      _httpclient.executeMethod(getMethod);
      assertEquals(HttpStatus.SC_NOT_FOUND, getMethod.getStatusCode());
      final AnyMap result = readResultMap(getMethod);
      assertTrue(result.getStringValue("message").startsWith(PipelineHandler.EXCEPTION_MESSAGE));
    } finally {
      getMethod.releaseConnection();
    }
  }

  /** read Http response and assert that it is a JSON map. */
  private AnyMap readResultMap(final HttpMethod method) throws IOException {
    final Any any = _anyReader.readJsonStream(method.getResponseBodyAsStream());
    if (any != null && any.isMap()) {
      return any.asMap();
    }
    fail("response must be an any map.");
    return null;
  }

}
