/*******************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Juergen Schumacher (empolis GmbH) - initial API and implementation
 *******************************************************************************/

package org.eclipse.smila.processing.bpel.test;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.processing.bpel.pipelet.SplitterPipelet;

/**
 * Test some basic operations of the workflow processor, not related to actual workflow processing.
 * 
 * @author jschumacher
 * 
 */
public class TestWorkflowProcessorService extends AWorkflowProcessorTest {

  private WorkflowProcessor _processor;

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _processor = getService(WorkflowProcessor.class);
    assertNotNull("no WorkflowProcessor service found.", _processor);
  }

  @Override
  protected String getPipelineName() {
    return "EchoPipeline";
  }

  /**
   * test listing of workflow names.
   */
  public void testWorkflowNames() throws Exception {
    final List<String> names = _processor.getWorkflowNames();
    assertNotNull(names);
    int count = 0;
    // count workflows from configuration only. workflows created in other tests have a name starting with "test".
    for (final String name : names) {
      if (!name.startsWith("test")) {
        count++;
      }
    }
    assertEquals(names.toString(), 11, count);
    assertTrue(names.contains("EchoPipeline"));
    assertTrue(names.contains("HelloWorldPipeline"));
    assertTrue(names.contains("SuperPipeline"));
  }

  /**
   * test reading of workflow definitions.
   */
  public void testGetWorkflowDefinition() throws Exception {
    final AnyMap noDefinition = _processor.getWorkflowDefinition("NoPipeline");
    assertNull(noDefinition);
    final AnyMap definitionMap = _processor.getWorkflowDefinition("EchoPipeline");
    assertNotNull(definitionMap);
    assertEquals("EchoPipeline", definitionMap.getStringValue(WorkflowProcessor.WORKFLOW_NAME));
    assertTrue(definitionMap.getBooleanValue(WorkflowProcessor.WORKFLOW_READONLY));
    final Any definition = definitionMap.get(WorkflowProcessor.WORKFLOW_DEFINITION);
    assertNotNull(definition);
    assertTrue(definition.isString());
    // Check if it is a SMILA BPEL pipeline.
    final String definitionString = definition.asValue().asString();
    assertNotNull(definitionString);
    assertFalse(definitionString.isEmpty());
    assertTrue(definitionString.contains("<process name=\"EchoPipeline\""));
    assertTrue(definitionString.contains("xmlns=\"http://docs.oasis-open.org/wsbpel/2.0/process/executable\""));
    assertTrue(definitionString.contains("targetNamespace=\"http://www.eclipse.org/smila/processor\""));
  }

  /** test creating a new workflow. */
  public void testSetWorkflowDefinition() throws Exception {
    final String pipelineName = "testSetWorkflowDefinition";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    assertNull(_processor.getWorkflowDefinition(pipelineName));
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertTrue(_processor.getWorkflowNames().contains(pipelineName));
    final AnyMap readDefinition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, bpel, readDefinition);
    assertEchoResult(pipelineName);
  }

  /** test creating a new workflow with readonly flag set and expecting it not to be stored. */
  public void testSetWorkflowDefinitionWithReadOnly() throws Exception {
    final String pipelineName = "testSetWorkflowDefinitionWithReadOnly";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    definition.put(WorkflowProcessor.WORKFLOW_READONLY, true);
    assertNull(_processor.getWorkflowDefinition(pipelineName));
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertTrue(_processor.getWorkflowNames().contains(pipelineName));
    final AnyMap readDefinition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, bpel, readDefinition);
  }

  /** test updating a new workflow. */
  public void testUpdateWorkflowDefinition() throws Exception {
    final String pipelineName = "testUpdateWorkflowDefinition";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertTrue(_processor.getWorkflowNames().contains(pipelineName));
    assertEchoResult(pipelineName);
    final String updateBpel = copyLocalHelloWorldPipeline(pipelineName);
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, updateBpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertTrue(_processor.getWorkflowNames().contains(pipelineName));
    final AnyMap readDefinition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, updateBpel, readDefinition);
    assertHelloWorldResult(pipelineName);
  }

  /** test creating a new workflow. */
  public void testDeleteWorkflowDefinition() throws Exception {
    final String pipelineName = "testDeleteWorkflowDefinition";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertTrue(_processor.getWorkflowNames().contains(pipelineName));
    assertEchoResult(pipelineName);
    _processor.deleteWorkflowDefinition(pipelineName);
    assertFalse(_processor.getWorkflowNames().contains(pipelineName));
    assertNull(_processor.getWorkflowDefinition(pipelineName));
  }

  /** test if trying to update a predefined workflow fails, and old definition stays active. */
  public void testErrorUpdatePredefinedWorkflow() throws Exception {
    final String pipelineName = TestHelloWorldPipeline.PIPELINE_NAME;
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
    assertHelloWorldResult(pipelineName);
  }

  /** test if trying to update an invalid definition fails, and old definition stays active. */
  public void testErrorInvalidWorkflow() throws Exception {
    final String pipelineName = "testErrorInvalidWorkflow";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    final String expectedTimestamp = definition.getStringValue(WorkflowProcessor.WORKFLOW_TIMESTAMP);
    assertEchoResult(pipelineName);
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, "this is no BPEL code.");
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
    final AnyMap readDefinition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, bpel, readDefinition);
    assertEquals(expectedTimestamp, readDefinition.getStringValue(WorkflowProcessor.WORKFLOW_TIMESTAMP));
    assertEchoResult(pipelineName);
  }

  /** test if trying to update an invalid definition fails, and old definition stays active. */
  public void testErrorInvalidNamespaceWorkflow() throws Exception {
    final String pipelineName = "testErrorInvalidNamespaceWorkflow";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    final String expectedTimestamp = definition.getStringValue(WorkflowProcessor.WORKFLOW_TIMESTAMP);
    assertEchoResult(pipelineName);
    final String badBpel =
      bpel.replace("targetNamespace=\"http://www.eclipse.org/smila/processor\"",
        "targetNamespace=\"http://www.eclipse.org/smila/processr\"");
    final AnyMap badDefinition = createWorkflowDefinition(pipelineName, badBpel);
    try {
      _processor.setWorkflowDefinition(pipelineName, badDefinition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
    final AnyMap readDefinition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, bpel, readDefinition);
    assertEquals(expectedTimestamp, readDefinition.getStringValue(WorkflowProcessor.WORKFLOW_TIMESTAMP));
    assertEchoResult(pipelineName);
  }

  /** test if trying to update again after an invalid definition works. */
  public void testUpdateAfterInvalidWorkflow() throws Exception {
    final String pipelineName = "testUpdateAfterInvalidWorkflow";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertEchoResult(pipelineName);
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, "this is no BPEL code.");
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
    final String bpel2 = copyLocalHelloWorldPipeline(pipelineName);
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, bpel2);
    _processor.setWorkflowDefinition(pipelineName, definition);
    final AnyMap read2Definition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, bpel2, read2Definition);
    assertHelloWorldResult(pipelineName);
  }

  /**
   * test if trying to update an undeployable definition (e.g. invoking external webservices) fails, and old definition
   * stays active.
   */
  public void testErrorUndeployableWorkflow() throws Exception {
    final String pipelineName = "testErrorUndeployableWorkflow";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    final String expectedTimestamp = definition.getStringValue(WorkflowProcessor.WORKFLOW_TIMESTAMP);
    assertEchoResult(pipelineName);
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, copyHelloWorldPipeline(pipelineName));
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
    final AnyMap readDefinition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, bpel, readDefinition);
    assertEquals(expectedTimestamp, readDefinition.getStringValue(WorkflowProcessor.WORKFLOW_TIMESTAMP));
    assertEchoResult(pipelineName);
  }

  /** test if trying to update again after an undeployable definition works. */
  public void testUpdateAfterUndeployableWorkflow() throws Exception {
    final String pipelineName = "testUpdateAfterUndeployableWorkflow";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertEchoResult(pipelineName);
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, copyHelloWorldPipeline(pipelineName));
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
    final String bpel2 = copyLocalHelloWorldPipeline(pipelineName);
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, bpel2);
    _processor.setWorkflowDefinition(pipelineName, definition);
    final AnyMap read2Definition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, bpel2, read2Definition);
    assertHelloWorldResult(pipelineName);
  }

  /** test if trying to update an invalid definition fails, and old definition stays active. */
  public void testErrorEmptyWorkflowCode() throws Exception {
    final String pipelineName = "testErrorEmptyWorkflow";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertEchoResult(pipelineName);
    definition.remove(WorkflowProcessor.WORKFLOW_DEFINITION);
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
    definition.put(WorkflowProcessor.WORKFLOW_DEFINITION, "");
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
    final AnyMap readDefinition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, bpel, readDefinition);
    assertEchoResult(pipelineName);
  }

  /** test if trying to update an invalid definition fails. */
  public void testErrorEmptyWorkflowName() throws Exception {
    final String pipelineName = "";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
    definition.remove(WorkflowProcessor.WORKFLOW_NAME);
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
  }

  /** test definition with differing names.. */
  public void testErrorWrongWorkflowName() throws Exception {
    final String pipelineName = "testErrorWrongWorkflowName";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    try {
      _processor.setWorkflowDefinition("somethingelse", definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
  }

  /** test definition with differing names.. */
  public void testErrorInvalidWorkflowName() throws Exception {
    final String pipelineName = "testError+/InvalidWorkflowName";
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    try {
      _processor.setWorkflowDefinition(pipelineName, definition);
      fail("must not work.");
    } catch (final ProcessingException ex) {
      System.out.println(ex);
    }
  }

  /**
   * test if processing works without any pause during update of an pipeline. Makes some time measurements to check if
   * the update does not slow down pipelines from other units, see System.out for timing results.
   */
  public void testProcessingDuringUpdate() throws Exception {
    final String pipelineName = "testProcessingDuringUpdate";
    // start with a pipeline as copy of Echo pipeline.
    final String bpel = copyEchoPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    _processor.setWorkflowDefinition(pipelineName, definition);
    assertEchoResult(pipelineName);
    final int numberOfThreads = 16;
    final ScheduledExecutorService executor = Executors.newScheduledThreadPool(numberOfThreads);
    try {
      final Throwable[] errors = new Throwable[numberOfThreads];
      final long[] sumRuntimes = new long[numberOfThreads];
      final long[] maxRuntimes = new long[numberOfThreads];
      final long[] numberOfInvokes = new long[numberOfThreads];
      for (int i = 0; i < numberOfThreads; i++) {
        final int index = i;
        executor.scheduleWithFixedDelay(new Runnable() {
          private boolean _updateDone;

          @Override
          public void run() {
            final boolean testUpdatedPipeline = index < numberOfThreads / 2;
            final String myPipelineName = testUpdatedPipeline ? pipelineName : TestEchoPipeline.PIPELINE_NAME;
            try {
              final String request = createBlackboardRecord("source", "key" + index);
              final long startTime = System.nanoTime();
              final String helloValue = executeAndGetWorkflowAttribute(myPipelineName, request);
              final long runtime = (System.nanoTime() - startTime) / 1000;
              sumRuntimes[index] += runtime;
              maxRuntimes[index] = Math.max(maxRuntimes[index], runtime);
              numberOfInvokes[index]++;
              if (testUpdatedPipeline) {
                if (_updateDone || !"SMILA".equals(helloValue)) {
                  assertEquals("Worker " + index + " while updateDone=" + _updateDone, "Hello SMILA", helloValue);
                  _updateDone = true;
                }
              } else {
                assertEquals("Worker " + index, "SMILA", helloValue);
              }
            } catch (final Throwable ex) {
              System.err.println("Error in worker " + index + ": " + ex);
              errors[index] = ex;
            }
          }
        }, 1, 1, TimeUnit.NANOSECONDS); // run as often as possible
      }
      Thread.sleep(1000); // let the workers get running.
      // redefine pipeline as copy of HelloWorld pipeline.
      final String newBpel = copyLocalHelloWorldPipeline(pipelineName);
      final AnyMap newDefinition = createWorkflowDefinition(pipelineName, newBpel);
      System.out.println("### START: UPDATING PIPELINE ###");
      final long startTime = System.nanoTime();
      _processor.setWorkflowDefinition(pipelineName, newDefinition);
      System.out.println("### DONE: UPDATING PIPELINE ### Time in us: " + (System.nanoTime() - startTime) / 1000);
      assertHelloWorldResult(pipelineName);
      Thread.sleep(1000); // let the workers run a bit more.
      // check if errors occurred in background workers.
      System.out.println("Max runtimes in us: " + Arrays.toString(maxRuntimes));
      System.out.println("Numbers of invocations: " + Arrays.toString(numberOfInvokes));
      for (int i = 0; i < numberOfThreads; i++) {
        System.out.println("Average runtime in worker " + i + " in us: " + sumRuntimes[i] / numberOfInvokes[i]);
        assertNull("Error in worker " + i + ": " + errors[i], errors[i]);
      }
    } finally {
      executor.shutdown();
      executor.awaitTermination(1, TimeUnit.SECONDS);
    }
  }

  /** create a new pipeline that uses calls a pipeline an already existing pipeline. */
  public void testCreateSuperPipeline() throws Exception {
    final String pipelineName = "testCreateSuperPipeline";
    final String bpel = copySuperPipeline(pipelineName);
    final AnyMap definition = createWorkflowDefinition(pipelineName, bpel);
    assertNull(_processor.getWorkflowDefinition(pipelineName));
    _processor.setWorkflowDefinition(pipelineName, definition);
    final AnyMap readDefinition = _processor.getWorkflowDefinition(pipelineName);
    assertWorkflowDefinition(pipelineName, bpel, readDefinition);
    assertSuperResult(pipelineName);
  }

  /** create blackboard record with "workflow-attribute" set to "SMILA". */
  @Override
  protected String createBlackboardRecord(final String source, final String key) throws BlackboardAccessException {
    final String id = super.createBlackboardRecord(source, key);
    getBlackboard().getMetadata(id).put("workflow-attribute", "SMILA");
    return id;
  }

  /** check that the named pipeline behaves like the EchoPipeline. */
  private void assertEchoResult(final String pipelineName) throws Exception {
    final String request = createBlackboardRecord("source", "key");
    final AnyMap expectedMetadata = DataFactory.DEFAULT.cloneAnyMap(getBlackboard().getMetadata(request));
    final String[] result = getProcessor().process(pipelineName, getBlackboard(), new String[] { request });
    assertEquals(1, result.length);
    assertEquals(request, result[0]);
    assertEquals(expectedMetadata, getBlackboard().getMetadata(request));
  }

  /** check that the named pipeline behaves like the LocalHelloWorldPipeline. */
  private void assertHelloWorldResult(final String pipelineName) throws Exception {
    final String request = createBlackboardRecord("source", "key");
    final String value = executeAndGetWorkflowAttribute(pipelineName, request);
    assertEquals("Hello SMILA", value);
  }

  /** check that the named pipeline behaves like the SuperPipeline. */
  void assertSuperResult(final String pipelineName) throws Exception {
    final String request = createBlackboardRecord("source", "key");
    final String[] result = getProcessor().process(pipelineName, getBlackboard(), new String[] { request });
    assertEquals(SplitterPipelet.SPLIT_FACTOR, result.length);
    for (final String element : result) {
      assertTrue(element.startsWith(request));
      assertTrue(request.length() < element.length());
      assertEquals(2, element.split(SplitterPipelet.FRAGMENT_MARKER).length);
      final AnyMap metadata = getBlackboard().getMetadata(element);
      assertEquals("SMILA", metadata.getStringValue("workflow-attribute"));
      assertEquals("value1", metadata.getStringValue("single-config-value"));
      assertEquals("value-a", metadata.getSeq("multi-config-value").getStringValue(0));
      assertEquals("value-b", metadata.getSeq("multi-config-value").getStringValue(1));
      assertEquals("sub-value", metadata.getMap("sub-config").getStringValue("sub-config-value"));
    }
  }

  /** execute a pipeline and get a single string value from "workflow-attribute" in the result record. */
  private String executeAndGetWorkflowAttribute(final String pipelineName, final String request)
    throws ProcessingException, BlackboardAccessException {
    final String[] result = getProcessor().process(pipelineName, getBlackboard(), new String[] { request });
    assertEquals(1, result.length);
    assertEquals(request, result[0]);
    final Any hello = getBlackboard().getMetadata(result[0]).get("workflow-attribute");
    assertNotNull(hello);
    assertNotNull(hello.isValue());
    final String value = hello.asValue().asString();
    return value;
  }
}
