/***********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - implementation
 **********************************************************************************************************************/
package org.eclipse.smila.processing.bpel.internal;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Properties;

import javax.xml.namespace.QName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.ode.ODEConfigProperties;
import org.eclipse.smila.ode.ODEServer;
import org.eclipse.smila.ode.ODEServerException;
import org.eclipse.smila.ode.WebServiceContextFactory;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.processing.bpel.BpelEngine;
import org.eclipse.smila.processing.bpel.util.BpelConstants;
import org.eclipse.smila.processing.bpel.util.ConfigurationHelper;
import org.w3c.dom.Element;

/** wrapper for {@link ODEServer} and {@link DeploymentManager}. */
public class OdeBpelEngine implements BpelEngine {

  /** BPEL server. */
  private ODEServer _odeServer;

  /** Workflow deployment manager for the ODEServer. */
  private DeploymentManager _deploymentManager;

  /** service providing an extension bundle for ODE. */
  private ExtensionBundleProvider _extensionProvider;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** initialize engine. */
  protected void activate() {
    try {
      final Properties properties = ConfigurationHelper.readConfiguration();
      final ODEConfigProperties odeConfig =
        new ODEConfigProperties(properties, ConfigurationHelper.PROP_PREFIX_ODE);
      final WebServiceContextFactory processingContext = new WebServiceContextFactory();
      _odeServer = new ODEServer(odeConfig, processingContext);
      _odeServer.registerExtensionBundle(_extensionProvider.getExtensionBundle());
      _deploymentManager = new DeploymentManager();
    } catch (final Exception ex) {
      _log.error("Start of BPEL workflow service failed: Unknown fatal error. "
        + "Service is non-functional, please fix problem and restart bundle", ex);
    }
  }

  /** bind DS service reference. */
  public void setExtensionProvider(final ExtensionBundleProvider extensionProvider) {
    _extensionProvider = extensionProvider;
  }

  /** unbind DS service reference. */
  public void unsetExtensionProvider(final ExtensionBundleProvider extensionProvider) {
    if (_extensionProvider == extensionProvider) {
      _extensionProvider = null;
    }
  }

  /** shutdown engine. */
  protected void deactivate() {
    _odeServer.shutdown();
  }

  @Override
  public Element invoke(final String workflowName, final Element message) throws ODEServerException {
    final QName processId = getProcessId(workflowName);
    return _odeServer.invoke(processId, BpelConstants.OPERATION_NAME, message);
  }

  @Override
  public boolean isPredefinedWorkflow(final String workflowName) {
    return _deploymentManager.isPredefinedWorkflow(workflowName);
  }

  @Override
  public boolean isCustomWorkflow(final String workflowName) {
    return _deploymentManager.isCustomWorkflow(workflowName);
  }

  @Override
  public AnyMap getWorkflow(final String workflowName) throws IOException {
    final AnyMap resultMap = DataFactory.DEFAULT.createAnyMap();
    final QName processId = getProcessId(workflowName);
    final String bpelContent = _odeServer.getBpelDocument(processId);
    if (bpelContent != null) {
      final Value bpel = DataFactory.DEFAULT.createStringValue(bpelContent);
      resultMap.put(WorkflowProcessor.WORKFLOW_NAME, workflowName);
      resultMap.put(WorkflowProcessor.WORKFLOW_READONLY, true);
      resultMap.put(WorkflowProcessor.WORKFLOW_DEFINITION, bpel);
      return resultMap;
    }
    return null;
  }

  @Override
  public File validateWorkflow(final String workflowName, final AnyMap workflowDefinition)
    throws ProcessingException {
    return _deploymentManager.validateWorkflow(workflowName, workflowDefinition);
  }

  @Override
  public QName deployWorkflowDir(final String workflowName, final File deploymentDirectory)
    throws ProcessingException {
    return _deploymentManager.deployWorkflowDir(workflowName, deploymentDirectory, _odeServer);
  }

  @Override
  public void undeployWorkflow(final String workflowName) throws ProcessingException {
    _deploymentManager.undeployWorkflow(workflowName, _odeServer);
  }

  @Override
  public void deployWorkflow(final String workflowName, final AnyMap workflowDefinition) throws ProcessingException {
    _deploymentManager.deployWorkflow(workflowName, workflowDefinition, _odeServer);
  }

  @Override
  public Collection<String> deployPredefinedWorkflows(final String pipelineDirName) throws ProcessingException,
    IOException {
    final Collection<QName> processNames =
      _deploymentManager.deployPredefinedWorkflows(pipelineDirName, _odeServer);
    final Collection<String> workflowNames = new ArrayList<String>(processNames.size());
    for (final QName processName : processNames) {
      workflowNames.add(processName.getLocalPart());
    }
    return workflowNames;
  }

  /** @return process id for workflowname. */
  private QName getProcessId(final String workflowName) {
    return new QName(BpelConstants.NAMESPACE_PROCESSOR, workflowName);
  }
}
