/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.processing.pipelets.test;

import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.custommonkey.xmlunit.Diff;
import org.custommonkey.xmlunit.XMLUnit;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.pipelets.ATransformationPipelet;
import org.eclipse.smila.processing.pipelets.SourceType;
import org.eclipse.smila.processing.util.ProcessingConstants;

/**
 * Base class for transformation pipelet tests.
 * 
 * @author aweber
 */
public abstract class ATransformationPipeletTest extends APipeletTest {

  /**
   * The CompareType decides how to compare xml's.
   */
  public enum CompareType {
    /**
     * compare for exact equality.
     */
    EQUAL,
    /**
     * compare for xml equality.
     */
    XML_DIFF
  }

  /**
   * @return configuration with ATTACHMENT as input-/outputType
   */
  protected AnyMap createAttachmentsConfiguration() {
    final AnyMap configuration = getBlackboard().getDataFactory().createAnyMap();
    configuration.put(ATransformationPipelet.PROP_INPUT_TYPE, SourceType.ATTACHMENT.name());
    configuration.put(ATransformationPipelet.PROP_OUTPUT_TYPE, SourceType.ATTACHMENT.name());
    configuration.put(ATransformationPipelet.PROP_INPUT_NAME, "in");
    configuration.put(ATransformationPipelet.PROP_OUTPUT_NAME, "out");
    return configuration;
  }

  /**
   * @return configuration with ATTRIBUTE as input-/outputType
   */
  protected AnyMap createAttributesConfiguration() {
    final AnyMap configuration = getBlackboard().getDataFactory().createAnyMap();
    configuration.put(ATransformationPipelet.PROP_INPUT_TYPE, SourceType.ATTRIBUTE.name());
    configuration.put(ATransformationPipelet.PROP_OUTPUT_TYPE, SourceType.ATTRIBUTE.name());
    configuration.put(ATransformationPipelet.PROP_INPUT_NAME, "in");
    configuration.put(ATransformationPipelet.PROP_OUTPUT_NAME, "out");
    return configuration;
  }

  /**
   * @return configuration with ATTACHMENT as inputType and ATTRIBUTE as outputType
   */
  protected AnyMap createAttachmentsAttributesConfiguration() {
    final AnyMap configuration = getBlackboard().getDataFactory().createAnyMap();
    configuration.put(ATransformationPipelet.PROP_INPUT_TYPE, SourceType.ATTACHMENT.name());
    configuration.put(ATransformationPipelet.PROP_OUTPUT_TYPE, SourceType.ATTRIBUTE.name());
    configuration.put(ATransformationPipelet.PROP_INPUT_NAME, "in");
    configuration.put(ATransformationPipelet.PROP_OUTPUT_NAME, "out");
    return configuration;
  }

  /**
   * @return configuration with ATTRIBUTE as inputType and ATTACHMENT as outputType
   */
  protected AnyMap createAttributesAttachmentsConfiguration() {
    final AnyMap configuration = getBlackboard().getDataFactory().createAnyMap();
    configuration.put(ATransformationPipelet.PROP_INPUT_TYPE, SourceType.ATTRIBUTE.name());
    configuration.put(ATransformationPipelet.PROP_OUTPUT_TYPE, SourceType.ATTACHMENT.name());
    configuration.put(ATransformationPipelet.PROP_INPUT_NAME, "in");
    configuration.put(ATransformationPipelet.PROP_OUTPUT_NAME, "out");
    return configuration;
  }

  /**
   * tests error handling in case of single record failures for input files.
   */
  protected void doRobustnessTestFileInput(final ATransformationPipelet pipelet, final SourceType sourceType,
    final String inputFile, final AnyMap config) throws Exception {
    final String inputString =
      IOUtils.toString(new FileInputStream(inputFile), ATransformationPipelet.ENCODING_ATTACHMENT);
    doRobustnessTestStringInput(pipelet, sourceType, inputString, config);
  }

  /**
   * tests error handling in case of single record failures for input strings.
   */
  protected void doRobustnessTestStringInput(final ATransformationPipelet pipelet, final SourceType sourceType,
    final String inputString, final AnyMap config) throws Exception {
    doRobustnessTest(pipelet, sourceType, inputString, config, true, true);
    doRobustnessTest(pipelet, sourceType, inputString, config, false, true);
  }

  /**
   * tests error handling in case of single record failures for input strings.
   */
  protected void doRobustnessTest(final ATransformationPipelet pipelet, final SourceType sourceType,
    final String inputString, final AnyMap config, boolean failOnError, boolean checkResults) throws Exception {

    if (failOnError) {
      config.put(ProcessingConstants.KEY_FAIL_ON_ERROR, "true");
    } else {
      config.put(ProcessingConstants.KEY_FAIL_ON_ERROR, "false");
    }
    final ParameterAccessor paramAccessor = new ParameterAccessor(getBlackboard(), config);
    final String inputName = pipelet.getInputName(paramAccessor);

    // prepare three records as testdata, one of them will be prepared to fail
    final int noOfRecords = 3;
    final List<String> recordIds = new ArrayList<String>();
    for (int i = 1; i <= noOfRecords; i++) {
      final String id = createBlackboardRecord("testSource", String.valueOf(i));
      recordIds.add(id);
      getBlackboard().getMetadata(id).put("_parameters", DataFactory.DEFAULT.createAnyMap());
      if (sourceType == SourceType.ATTACHMENT) {
        getBlackboard().setAttachment(id, inputName,
          inputString.getBytes(ATransformationPipelet.ENCODING_ATTACHMENT));
      } else {
        getBlackboard().getMetadata(id).put(inputName, inputString);
      }
      if (i == 2) { // prepare error record
        getBlackboard().getMetadata(id).getMap("_parameters")
          .put(ATransformationPipelet.PROP_INPUT_TYPE, "invalid_input_type");
      }
    }

    // execute
    if (failOnError) {
      try {
        pipelet.process(getBlackboard(), recordIds.toArray(new String[recordIds.size()]));
        fail("failOnError=true -> exception should be thrown");
      } catch (final ProcessingException e) {
        ; // expected
      }
    } else {
      final String[] result = pipelet.process(getBlackboard(), recordIds.toArray(new String[recordIds.size()]));
      if (checkResults) {
        final List<String> resultIds = Arrays.asList(result);
        if (ProcessingConstants.DROP_ON_ERROR_DEFAULT) {
          assertEquals(noOfRecords - 1, resultIds.size());
          recordIds.remove(1);
          assertEquals(recordIds, resultIds);
        } else {
          assertEquals(noOfRecords, resultIds.size());
          assertEquals(recordIds, resultIds);
        }
      }
    }
  }

  /**
   * @param pipelet
   *          the pipelet to test
   * @param sourceType
   *          the source type (ATTRIBUTE/ATTACHMENT)
   * @param inputFile
   *          the file containing the test input
   * @param expectedFile
   *          the file containing the expected test output
   * @param compareType
   *          whether to test for equality or use XML Diff to compare result with expected
   * @throws Exception
   *           error
   */
  protected void doInputOutputTest(final ATransformationPipelet pipelet, final SourceType sourceType,
    final String inputFile, final String expectedFile, final CompareType compareType, final AnyMap config)
    throws Exception {

    final ParameterAccessor paramAccessor = new ParameterAccessor(getBlackboard(), config);
    final String inputName = pipelet.getInputName(paramAccessor);
    final String outputName = pipelet.getOutputName(paramAccessor);

    // prepare test data
    final String id = createBlackboardRecord("testSource", "testId");
    final String[] recordIds = new String[] { id };
    if (sourceType == SourceType.ATTACHMENT) {
      getBlackboard().setAttachment(id, inputName, IOUtils.toByteArray(new FileInputStream(inputFile)));
    } else {
      getBlackboard().getMetadata(id).put(inputName,
        IOUtils.toString(new FileInputStream(inputFile), ATransformationPipelet.ENCODING_ATTACHMENT));
    }

    // execute
    pipelet.process(getBlackboard(), recordIds);

    // check result
    String xmlResult = null;
    if (sourceType == SourceType.ATTACHMENT) {
      assertTrue(getBlackboard().hasAttachment(id, outputName));
      final byte[] result = getBlackboard().getAttachmentAsBytes(id, outputName);
      assertNotNull(result);
      xmlResult = new String(result, ATransformationPipelet.ENCODING_ATTACHMENT);
    } else {
      assertTrue(getBlackboard().getMetadata(id).containsKey(outputName));
      xmlResult = getBlackboard().getMetadata(id).getStringValue(outputName);
    }
    assertNotNull(xmlResult);
    final String expected =
      removeBOM(IOUtils.toString(new FileInputStream(expectedFile), ATransformationPipelet.ENCODING_ATTACHMENT));
    if (compareType == CompareType.XML_DIFF) {
      final Diff diff = XMLUnit.compareXML(expected, xmlResult);
      assertTrue(diff.identical());
    } else {
      assertEquals(expected, xmlResult);
    }
  }
}
