/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH.
 * All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Tobias Liefke - initial API and implementation
 *******************************************************************************/

package org.eclipse.smila.processing.pipelets.test;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.pipelets.FileWriterPipelet;
import org.eclipse.smila.processing.util.ProcessingConstants;

/**
 * Test class for FileWriter pipelet.
 * 
 * @author Tobias Liefke
 */
public class TestFileWriterPipelet extends APipeletTest {

  /** Test content for the test file. */
  private static final String TEST_CONTENT = "blabla\nblub blub\n\u00e4\u00f6\n"
    + "Lorem ipsum dolor sit amet, consectetur adipisici elit, "
    + "sed eiusmod tempor incidunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, "
    + "quis nostrud exercitation ullamco laboris nisi ut aliquid ex ea commodi consequat. "
    + "Quis aute iure reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. "
    + "Excepteur sint obcaecat cupiditat non proident, sunt in culpa qui officia deserunt "
    + "mollit anim id est laborum.\nDuis autem vel eum iriure dolor in hendrerit in vulputate velit "
    + "esse molestie consequat, vel illum dolore eu feugiat nulla facilisis at vero eros et "
    + "accumsan et iusto odio dignissim qui blandit praesent luptatum zzril delenit augue duis "
    + "dolore te feugait nulla facilisi. Lorem ipsum dolor sit amet, consectetuer adipiscing elit, "
    + "sed diam nonummy nibh euismod tincidunt ut laoreet dolore magna aliquam erat volutpat.\n"
    + "Ut wisi enim ad minim veniam, quis nostrud exerci tation ullamcorper suscipit lobortis nisl "
    + "ut aliquip ex ea commodo consequat. Duis autem vel eum iriure dolor in hendrerit in "
    + "vulputate velit esse molestie consequat, vel illum dolore eu feugiat nulla facilisis at vero "
    + "eros et accumsan et iusto odio dignissim qui blandit praesent luptatum zzril delenit augue "
    + "duis dolore te feugait nulla facilisi.\n";

  /** Name of the attachment to use for testing. */
  private static final String CONTENT_ATTACHMENT = "content";

  /** The file writer pipelet. */
  private FileWriterPipelet _fileWriterPipelet;

  /**
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#setUp()
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _fileWriterPipelet = new FileWriterPipelet();

    final AnyMap configuration = _blackboard.getDataFactory().createAnyMap();
    configuration.put(FileWriterPipelet.PROPERTY_PATH_ATTRIBUTE, "path");
    configuration.put(FileWriterPipelet.PROPERTY_CONTENT_ATTACHMENT, CONTENT_ATTACHMENT);
    configuration.put(ProcessingConstants.KEY_FAIL_ON_ERROR, true);
    _fileWriterPipelet.configure(configuration);
  }

  /**
   * Test writing to a new file.
   * 
   * @throws Exception
   *           if the test fails
   */
  public void testWritingFile() throws Exception {
    final File testFile = File.createTempFile(getName(), ".tmp");
    try {
      final Record record = _blackboard.getRecord("writeId", Get.AUTO_CREATE);
      final AnyMap metaData = record.getMetadata();
      metaData.put("path", testFile.getCanonicalPath());
      _blackboard.setRecord(record);
      byte[] bytes = TEST_CONTENT.getBytes("UTF-8");
      final String[] recordIds = new String[] { record.getId() };

      // Check if file is created
      _blackboard.setAttachment(record.getId(), CONTENT_ATTACHMENT, bytes);
      String[] result = _fileWriterPipelet.process(_blackboard, recordIds);
      assertNotNull(result);
      assertEquals(recordIds.length, result.length);
      FileInputStream fileStream = new FileInputStream(testFile);
      try {
        assertTrue("File does not contain the attachment",
          IOUtils.contentEquals(new ByteArrayInputStream(bytes), fileStream));
      } finally {
        IOUtils.closeQuietly(fileStream);
      }

      // Check if file is overwritten
      bytes = new byte[] { 0x12, 0x34, 0x56, 0x78 };
      _blackboard.setAttachment(record.getId(), CONTENT_ATTACHMENT, bytes);
      result = _fileWriterPipelet.process(_blackboard, recordIds);
      assertNotNull(result);
      assertEquals(recordIds.length, result.length);
      fileStream = new FileInputStream(testFile);
      try {
        assertTrue("File does not contain the attachment",
          IOUtils.contentEquals(new ByteArrayInputStream(bytes), fileStream));
      } finally {
        IOUtils.closeQuietly(fileStream);
      }

      // Check if append is working
      metaData.getMap(ParameterAccessor.DEFAULT_PARAMETERS_ATTRIBUTE, true).put(FileWriterPipelet.PROPERTY_APPEND,
        Boolean.TRUE);
      bytes = new byte[] { (byte) 0x9A, (byte) 0xBC, (byte) 0xDE };
      _blackboard.setAttachment(record.getId(), CONTENT_ATTACHMENT, bytes);
      result = _fileWriterPipelet.process(_blackboard, recordIds);
      assertNotNull(result);
      assertEquals(recordIds.length, result.length);
      fileStream = new FileInputStream(testFile);
      try {
        assertTrue(
          "File does not contain the concatenated attachments",
          IOUtils.contentEquals(new ByteArrayInputStream(new byte[] { 0x12, 0x34, 0x56, 0x78, (byte) 0x9A,
            (byte) 0xBC, (byte) 0xDE }), fileStream));
      } finally {
        IOUtils.closeQuietly(fileStream);
      }
    } finally {
      FileUtils.deleteQuietly(testFile);
    }
  }

}
