/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.processing.pipelets.xmlprocessing.test;

import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.validation.InvalidRecordException;
import org.eclipse.smila.jobmanager.WorkflowRunInfo;
import org.eclipse.smila.processing.pipelets.SourceType;
import org.eclipse.smila.processing.pipelets.test.ATransformationPipeletTest;
import org.eclipse.smila.processing.pipelets.xmlprocessing.XmlSplitterPipelet;

/**
 * Test for {@link XmlSplitterPipelet} class.
 * 
 * @author aweber
 */
public class TestXmlSplitterPipelet extends ATransformationPipeletTest {

  /** Constant for the xml document to split. */
  private static final String SPLIT_XML = "./configuration/data/split.xml";

  /** test splitting of single xml file in three records with input attribute. */
  public void testAttributeInput() throws Exception {
    final AnyMap configuration = createAttributesConfiguration();
    final String id = createBlackboardRecord("testSource", "testAttributes");
    getBlackboard().getMetadata(id).put("in", SPLIT_XML);
    doTest(configuration, id);
  }

  /** test splitting of single xml file in three records with input attachment. */
  public void testAttachmentInput() throws Exception {
    final AnyMap configuration = createAttachmentsAttributesConfiguration();
    final String id = createBlackboardRecord("testSource", "testAttachments");
    getBlackboard().setAttachment(id, "in", IOUtils.toByteArray(new FileInputStream(SPLIT_XML)));
    doTest(configuration, id);
  }

  /** tests error handling in case of single record failures. */
  public void testRobustness() throws Exception {
    final XmlSplitterPipelet pipelet = new XmlSplitterPipelet();
    final BulkBuilderStub bulkBuilder = new BulkBuilderStub();
    pipelet.setBulkbuilderService(bulkBuilder);
    final AnyMap configuration = createAttributesConfiguration();
    configuration.put(XmlSplitterPipelet.PROP_BEGIN_TAG_NAME, "document");
    configuration.put(XmlSplitterPipelet.PROP_END_TAG_NAME, "document");
    configuration.put(XmlSplitterPipelet.PROP_KEY_TAG_NAME, "docId");
    configuration.put(XmlSplitterPipelet.PROP_ID_SEPARATOR, "#");
    configuration.put(XmlSplitterPipelet.PROP_JOB_NAME, "job");
    pipelet.configure(configuration);
    doRobustnessTest(pipelet, SourceType.ATTRIBUTE, SPLIT_XML, configuration, true, false);
    doRobustnessTest(pipelet, SourceType.ATTRIBUTE, SPLIT_XML, configuration, false, false);
  }

  /** do test and check the results. */
  private void doTest(final AnyMap configuration, final String id) throws Exception {
    final XmlSplitterPipelet pipelet = new XmlSplitterPipelet();
    final BulkBuilderStub bulkBuilder = new BulkBuilderStub();
    pipelet.setBulkbuilderService(bulkBuilder);
    final String[] recordIds = new String[] { id };

    configuration.put(XmlSplitterPipelet.PROP_BEGIN_TAG_NAME, "document");
    configuration.put(XmlSplitterPipelet.PROP_END_TAG_NAME, "document");
    configuration.put(XmlSplitterPipelet.PROP_KEY_TAG_NAME, "docId");
    configuration.put(XmlSplitterPipelet.PROP_ID_SEPARATOR, "#");
    configuration.put(XmlSplitterPipelet.PROP_JOB_NAME, "job");
    pipelet.configure(configuration);

    pipelet.process(getBlackboard(), recordIds);

    final Map<String, Record> resultRecords = bulkBuilder.getRecords("job");
    assertEquals(3, resultRecords.size());
    final List<String> outs = new ArrayList<String>();
    for (final Record r : resultRecords.values()) {
      final String outXml = r.getMetadata().getStringValue("out");
      assertTrue(outXml.matches("<document>(.|\\s)*</document>"));
      assertTrue(outXml.contains("<docId>"));
      assertTrue(outXml.contains("<title>"));
      assertTrue(outXml.contains("<text>"));
      assertFalse(outs.contains(outXml));
      outs.add(outXml);
    }
  }

  private static final class BulkBuilderStub implements BulkbuilderService {
    private Map<String, Map<String, Record>> _records = new HashMap<String, Map<String, Record>>();

    /** @return records */
    public Map<String, Record> getRecords(final String jobName) {
      return _records.get(jobName);
    }

    public WorkflowRunInfo addRecord(final String jobName, final Record record) throws BulkbuilderException,
      InvalidRecordException {
      Map<String, Record> jobRecords = _records.get(jobName);
      if (jobRecords == null) {
        jobRecords = new HashMap<String, Record>();
        _records.put(jobName, jobRecords);
      }
      jobRecords.put(record.getId(), record);

      return new WorkflowRunInfo(jobName, "dummyJobId", "dummyWorkflowId");
    }

    public WorkflowRunInfo deleteRecord(final String jobName, final Record record) throws BulkbuilderException,
      InvalidRecordException {
      Map<String, Record> jobRecords = _records.get(jobName);
      if (jobRecords != null) {
        jobRecords.remove(record.getId());
      }
      return new WorkflowRunInfo(jobName, "dummyJobId", "dummyWorkflowId");
    }

    public WorkflowRunInfo commitJob(final String jobName) throws BulkbuilderException {
      // Nothing to do
      return new WorkflowRunInfo(jobName, "dummyJobId", "dummyWorkflowId");
    }

    public WorkflowRunInfo addToMicroBulk(final String jobName, final Record record, final String microBulkId)
      throws BulkbuilderException, InvalidRecordException {
      // Nothing to do
      return new WorkflowRunInfo(jobName, "dummyJobId", "dummyWorkflowId");
    }

    public WorkflowRunInfo finishMicroBulk(final String jobName, final String microBulkId)
      throws BulkbuilderException {
      // Nothing to do
      return new WorkflowRunInfo(jobName, "dummyJobId", "dummyWorkflowId");
    }

    public void removeMicroBulk(final String microBulkId) {
      // Nothing to do
    }
  }

}
