/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation 
 **********************************************************************************************************************/

package org.eclipse.smila.restapi;

import java.util.Collection;
import java.util.HashSet;

import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.workermanager.WorkerManager;
import org.eclipse.smila.zookeeper.ZooKeeperService;

/**
 * Implements the handling of HTTP state debug requests.
 */
public class DebugHandler extends JsonRequestHandler {
  /** WorkerManager reference. */
  private WorkerManager _workerManager;

  /** DefinitionPersistence reference. */
  private DefinitionPersistence _defPersistence;

  /** ClusterConfigService. */
  private ClusterConfigService _clusterService;

  /** zookeeper service. */
  private ZooKeeperService _zkService;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) {
    final AnyMap result = FACTORY.createAnyMap();
    if (_workerManager != null) {
      result.put("workerManager", buildWorkerSection());
    }
    if (_zkService != null && _clusterService != null) {
      result.put("zookeeper", buildZookeeperSection());
    }
    return result;
  }

  /** @return Any, representing worker(manager) section. */
  private AnyMap buildWorkerSection() {
    final AnyMap workerInfo = _workerManager.getInfo();
    final Collection<String> workerNames = _defPersistence.getWorkers();
    for (String workerName : workerNames) {
      // "bulkbuilder" is the only worker that is not registered in workermanager
      if (!_workerManager.hasWorker(workerName) && !"bulkbuilder".equals(workerName)) {
        final AnyMap missingWorker = DataFactory.DEFAULT.createAnyMap();
        missingWorker.put("name", workerName);
        missingWorker
          .put(
            "WARNING",
            "Worker is not registered in WorkerManager. Maybe worker OSGI service isn't started or worker's name setting is inconsistent.");
        workerInfo.getSeq("workers").add(missingWorker);
      }
    }
    return workerInfo;
  }

  /**
   * Build "zookeeper" section.
   * 
   * @return Any, representing zookeeper section
   */
  private AnyMap buildZookeeperSection() {
    try {
      return _zkService.getServerState();
    } catch (final Exception ex) {
      return exceptionToAny(new IllegalArgumentException("Error creating zookeeper section", ex));
    }
  }

  /**
   * {@inheritDoc}
   * 
   * GET is currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /**
   * convert an exception to an any object.
   * 
   * @param e
   *          exception to convert
   * @return any representation of exception
   */
  private AnyMap exceptionToAny(final Throwable e) {
    try {
      final AnyMap error = FACTORY.createAnyMap();
      error.put("error", exceptionToAny(e, new HashSet<String>()));
      return error;
    } catch (final Exception drecksEx) {
      throw new IllegalArgumentException("Error converting exception \"" + e.toString() + "\"", drecksEx);
    }
  }

  /**
   * convert an exception to an any object. stop in stacktrace printing when hitting known lines again.
   * 
   * @param e
   *          exception to convert
   * @param visitedLines
   *          lines that have been added to stacktraces before.
   * @return any representation of exception
   * @throws Exception
   *           creating the object.
   */
  private AnyMap exceptionToAny(final Throwable e, final Collection<String> visitedLines) throws Exception {
    final AnyMap any = FACTORY.createAnyMap();
    any.put("type", e.getClass().getName());
    if (e.getMessage() != null) {
      any.put("message", e.getMessage());
    }
    final AnySeq st = FACTORY.createAnySeq();
    for (final StackTraceElement stElement : e.getStackTrace()) {
      final String line = stElement.toString();
      st.add(line);
      if (!visitedLines.add(line)) {
        st.add("...");
        break;
      }
    }
    any.put("at", st);
    if (e.getCause() != null && e.getCause() != e) {
      any.put("causedBy", exceptionToAny(e.getCause(), visitedLines));
    }
    return any;
  }

  /** sets referenced OSGI service */
  public void setWorkerManager(final WorkerManager wm) {
    _workerManager = wm;
  }

  /** unsets referenced OSGI service */
  public void unsetWorkerManager(final WorkerManager wm) {
    if (_workerManager == wm) {
      _workerManager = null;
    }
  }

  /** sets referenced OSGI service */
  public void setClusterConfigService(final ClusterConfigService clusterConfigService) {
    _clusterService = clusterConfigService;
  }

  /** unsets referenced OSGI service */
  public void unsetClusterConfigService(final ClusterConfigService clusterConfigService) {
    if (_clusterService == clusterConfigService) {
      _clusterService = null;
    }
  }

  /** sets referenced OSGI service */
  public void setZooKeeperService(final ZooKeeperService zkService) {
    _zkService = zkService;
  }

  /** unsets referenced OSGI service */
  public void unsetZooKeeperService(final ZooKeeperService zkService) {
    if (_zkService == zkService) {
      _zkService = null;
    }
  }

  /** sets referenced OSGI service */
  public void setDefinitionPersistence(final DefinitionPersistence defPers) {
    _defPersistence = defPers;
  }

  /** unsets referenced OSGI service */
  public void unsetDefinitionPersistence(final DefinitionPersistence defPers) {
    if (_defPersistence == defPers) {
      _defPersistence = null;
    }
  }

}
