/*******************************************************************************
 * Copyright (c) 2009 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (empolis GmbH) - initial API and implementation
 *******************************************************************************/

package org.eclipse.smila.search.api.test;

import java.util.Arrays;

import junit.framework.TestCase;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.blackboard.impl.BlackboardFactoryImpl;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.search.api.QueryConstants.SortOrder;
import org.eclipse.smila.search.api.helper.QueryBuilder;
import org.eclipse.smila.search.api.helper.QueryParameterAccessor;
import org.eclipse.smila.search.api.helper.ResultBuilder;
import org.eclipse.smila.search.api.helper.ResultItemAccessor;

/**
 * test query building and reading.
 * 
 * @author jschumacher
 * 
 */
public class BuilderAndAccessorTest extends TestCase {

  final Blackboard _blackboard = new BlackboardFactoryImpl().createTransientBlackboard();

  /**
   * test setting of parameters and accessing them.
   * 
   * @throws Exception
   *           test fails.
   */
  public void testParameters() throws Exception {
    final QueryBuilder builder = new QueryBuilder("pipeline");
    builder.setLanguage("en").setQuery("what am i thinking of").setMaxCount(2).setOffset(1).setThreshold(1.0);
    builder.setParameter("single", "value").addParameter("multi", "value1").addParameter("multi", "value2");
    builder.addSortBy("date", SortOrder.DESCENDING).addSortBy("size", SortOrder.ASCENDING);
    builder.addFacetByAttribute("author", 1).addFacetByAttribute("company", 2, "count", SortOrder.DESCENDING);
    final Record record = builder.getQuery();
    assertNotNull(record);
    final AnyMap parameters = record.getMetadata();
    assertNotNull(parameters);
    assertEquals("en", parameters.getStringValue(QueryConstants.LANGUAGE));
    assertEquals("what am i thinking of", parameters.getStringValue(QueryConstants.QUERY));
    assertEquals(2, parameters.getLongValue(QueryConstants.MAXCOUNT).intValue());
    assertEquals(1, parameters.getLongValue(QueryConstants.OFFSET).intValue());
    assertEquals(1.0, parameters.getDoubleValue(QueryConstants.THRESHOLD).doubleValue());
    assertEquals("value", parameters.getStringValue("single"));
    final AnySeq values = parameters.getSeq("multi");
    assertNotNull(values);
    assertEquals(2, values.size());
    assertEquals("value1", values.getStringValue(0));
    assertEquals("value2", values.getStringValue(1));
    final AnySeq sortBy = parameters.getSeq(QueryConstants.SORTBY);
    assertEquals(2, sortBy.size());
    assertEquals("date", sortBy.getMap(0).getStringValue(QueryConstants.ATTRIBUTE));
    assertEquals(SortOrder.DESCENDING.name().toLowerCase(), sortBy.getMap(0).getStringValue(QueryConstants.ORDER));
    assertEquals("size", sortBy.getMap(1).getStringValue(QueryConstants.ATTRIBUTE));
    assertEquals(SortOrder.ASCENDING.name().toLowerCase(), sortBy.getMap(1).getStringValue(QueryConstants.ORDER));
    final AnySeq facetBy = parameters.getSeq(QueryConstants.FACETBY);
    assertEquals(2, facetBy.size());
    assertEquals("author", facetBy.getMap(0).getStringValue(QueryConstants.ATTRIBUTE));
    assertEquals(1, facetBy.getMap(0).getLongValue(QueryConstants.MAXCOUNT).intValue());
    assertEquals("company", facetBy.getMap(1).getStringValue(QueryConstants.ATTRIBUTE));
    assertEquals(2, facetBy.getMap(1).getLongValue(QueryConstants.MAXCOUNT).intValue());
    assertEquals("count",
      facetBy.getMap(1).getMap(QueryConstants.SORTBY).getStringValue(QueryConstants.FACETBY_SORTCRITERION));
    assertEquals(SortOrder.DESCENDING.name().toLowerCase(), facetBy.getMap(1).getMap(QueryConstants.SORTBY)
      .getStringValue(QueryConstants.ORDER));
  }

  public void test_HightlightQueryBuilderAndParameterAccessor() throws Exception {
    final String id = "TestRecord";
    final Record record = _blackboard.getRecord(id, Get.NEW);
    final QueryBuilder qb = new QueryBuilder("TestWorkflow", record);
    final String[] highlight_attributes = { "title", "content" };
    qb.addHighlightByAttribute(highlight_attributes);
    final QueryParameterAccessor accessor = new QueryParameterAccessor(_blackboard, id);
    final AnySeq highlight = accessor.getHighlightConfig();
    assertEquals("title", highlight.getMap(0).getStringValue("attribute"));
    assertEquals("content", highlight.getMap(1).getStringValue("attribute"));
  }

  public void test_HighlightResultBuilderAndItemAccessor() throws Exception {
    final String id = "TestRecord";
    final Record record = _blackboard.getRecord(id, Get.NEW);
    final ResultBuilder rb = new ResultBuilder("TestWorkflow", record);
    final AnyMap resultItem1 = rb.addResultItem("Result1", 1.0);
    rb.addHighlightText(resultItem1, "title", "This is highlighted text.");
    final String[] highlighted_text =
      { "This is highlighted highlighted text too.", "This is another highlighted text." };
    final AnyMap resultItem2 = rb.addResultItem("Result2", 1.0);
    rb.addHighlightText(resultItem2, "content", Arrays.asList(highlighted_text));
    final ResultItemAccessor accessor1 = new ResultItemAccessor(0, resultItem1);
    final AnyMap _highlight1 = accessor1.getHighlight();
    assertEquals("This is highlighted text.", _highlight1.getMap("title").getStringValue("text"));
    final ResultItemAccessor accessor2 = new ResultItemAccessor(0, resultItem2);
    final AnyMap _highlight2 = accessor2.getHighlight();
    assertEquals("This is highlighted highlighted text too.", _highlight2.getMap("content").getSeq("text")
      .getStringValue(0));
    assertEquals("This is another highlighted text.", _highlight2.getMap("content").getSeq("text")
      .getStringValue(1));
  }

}
