/**
 * 
 */
package org.eclipse.smila.solr;

import java.io.UnsupportedEncodingException;
import java.util.UUID;

import junit.framework.TestCase;

import org.apache.commons.lang.StringUtils;
import org.apache.solr.common.params.FacetParams;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.Blackboard.Get;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.impl.BlackboardFactoryImpl;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.search.api.helper.QueryBuilder;
import org.eclipse.smila.solr.search.SolrQueryBuilder;
import org.eclipse.smila.solr.search.SolrQueryConverter;
import org.eclipse.smila.solr.search.SolrQueryParameterAccessor;
import org.eclipse.smila.solr.util.SolrQueryUtils;

/**
 * @author pwissel
 * 
 */
public class SolrQueryConverter_Test extends TestCase {

  private static final String WORKFLOW = "TestWorkflow";

  final Blackboard _blackboard = new BlackboardFactoryImpl().createTransientBlackboard();

  private Record createRecord(String... recordid) throws BlackboardAccessException {
    recordid[0] = StringUtils.defaultIfEmpty(recordid[0], "TestRecord_" + UUID.randomUUID());
    return _blackboard.getRecord(recordid[0], Get.NEW);
  }

  private void testSolrQueryConverter(String recordid, String expected) throws BlackboardAccessException,
    UnsupportedEncodingException {
    final SolrQueryParameterAccessor accessor = new SolrQueryParameterAccessor(_blackboard, recordid);
    final SolrQueryConverter converter = new SolrQueryConverter(accessor);

    final String query = converter.toSolrQuery(null).toString();
    final String actual = SolrQueryUtils.decodeQuery(query);
    assertEquals(expected, actual);
  }

  public void test_FacetSettings() throws Exception {
    final String recordid = "FacetSettings";
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FACET_TYPE, "facet.field");
    facetConfig.put(QueryConstants.ATTRIBUTE, "Extension");
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected = "facet=true&facet.field=Extension";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_FacetSettings_FacetName() throws Exception {
    final String recordid = "FacetSettings";
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FACET_TYPE, "facet.field");
    facetConfig.put(QueryConstants.ATTRIBUTE, "Extension");
    facetConfig.put(SolrConstants.FACET_NAME, "TestFacetName");
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected = "facet=true&facet.field={!key=TestFacetName}Extension";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_FacetSettings_Multiselect() throws Exception {
    final String recordid = "FacetSettings";
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FACET_TYPE, "facet.field");
    facetConfig.put(QueryConstants.ATTRIBUTE, "Extension");
    facetConfig.put(SolrConstants.FACET_MULTISELECT, true);
    final AnySeq filters = facetConfig.getSeq(QueryConstants.FILTER_ONEOF, true);
    filters.add("doc");
    filters.add("xsl");
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected =
      "facet=true&facet.field={!ex=facet_Extension}Extension&fq={!tag=facet_Extension}(Extension:doc)(Extension:xsl)";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_FacetSettings_CustomRanges() throws Exception {
    final String recordid = "FacetSettings_CustomRanges";
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FACET_TYPE, "custom.ranges");
    facetConfig.put(QueryConstants.ATTRIBUTE, "Size");
    facetConfig.put(SolrConstants.FACET_NAME, "FileSize");
    facetConfig.put(SolrConstants.FACET_MULTISELECT, true);
    final AnySeq ranges = facetConfig.getSeq(SolrConstants.FACET_RANGES, true);
    ranges.add("[* TO 999]");
    ranges.add("[1000 TO 9999999]");
    ranges.add("[9999999 TO *]");
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected =
      "facet=true&facet.query={!key=FileSize_0 ex=facet_FileSize}Size:[* TO 999]&facet.query={!key=FileSize_1 ex=facet_FileSize}Size:[1000 TO 9999999]&facet.query={!key=FileSize_2 ex=facet_FileSize}Size:[9999999 TO *]";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_FacetSettings_CustomRanges_WithFilter() throws Exception {
    final String recordid = "FacetSettings_CustomRanges_WithFilter";
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FACET_TYPE, "custom.ranges");
    facetConfig.put(QueryConstants.ATTRIBUTE, "Size");
    facetConfig.put(SolrConstants.FACET_NAME, "FileSize");
    facetConfig.put(SolrConstants.FACET_MULTISELECT, true);
    final AnySeq ranges = facetConfig.getSeq(SolrConstants.FACET_RANGES, true);
    ranges.add("[* TO 999]");
    ranges.add("[1000 TO 9999999]");
    ranges.add("[9999999 TO *]");
    final AnySeq filters = facetConfig.getSeq(QueryConstants.FILTER_ONEOF, true);
    filters.add("FileSize_1");
    filters.add("FileSize_2");
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected =
      "facet=true&facet.query={!key=FileSize_0 ex=facet_FileSize}Size:[* TO 999]&facet.query={!key=FileSize_1 ex=facet_FileSize}Size:[1000 TO 9999999]&facet.query={!key=FileSize_2 ex=facet_FileSize}Size:[9999999 TO *]&fq={!tag=facet_FileSize}(Size:[1000 TO 9999999])(Size:[9999999 TO *])";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_FacetSettings_MaxCount() throws Exception {
    final String recordid = "FacetSettings";
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FACET_TYPE, "facet.field");
    facetConfig.put(QueryConstants.ATTRIBUTE, "Extension");
    facetConfig.put(QueryConstants.MAXCOUNT, 10);
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected = "facet=true&facet.field=Extension&f.Extension.facet.limit=10";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_FacetSettings_Sort() throws Exception {
    final String recordid = "FacetSettings";
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FACET_TYPE, "facet.field");
    facetConfig.put(QueryConstants.ATTRIBUTE, "Extension");
    final AnyMap sortby = facetConfig.getMap(QueryConstants.SORTBY, true);
    sortby.put(QueryConstants.FACETBY_SORTCRITERION, "index");
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected = "facet=true&facet.field=Extension&f.Extension.facet.sort=index";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_FacetSettings_NativeParameters() throws Exception {
    final String recordid = "FacetSettings";
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FACET_TYPE, "facet.date");
    facetConfig.put(QueryConstants.ATTRIBUTE, "LastModifiedDate");
    final AnyMap nativeParameters = facetConfig.getMap(QueryConstants.NATIVE_PARAMETERS, true);
    nativeParameters.put(FacetParams.FACET_DATE_START, "NOW/DAY-5DAYS");
    nativeParameters.put(FacetParams.FACET_DATE_GAP, "+1DAY");
    nativeParameters.put(FacetParams.FACET_DATE_END, "NOW/DAY+1DAY");
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected =
      "facet=true&facet.date=LastModifiedDate&f.LastModifiedDate.facet.date.start=NOW/DAY-5DAYS&f.LastModifiedDate.facet.date.gap=+1DAY&f.LastModifiedDate.facet.date.end=NOW/DAY+1DAY";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_FacetSettings_Filters() throws Exception {
    final String recordid = "FacetSettings";
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FACET_TYPE, "facet.field");
    facetConfig.put(QueryConstants.ATTRIBUTE, "Extension");
    final AnySeq filters = facetConfig.getSeq(QueryConstants.FILTER_ONEOF, true);
    filters.add("doc");
    filters.add("xsl");
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected =
      "facet=true&facet.field=Extension&fq={!tag=facet_Extension}(Extension:doc)(Extension:xsl)";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_Filters_GroupName() throws Exception {
    final String recordid = getName();
    final Record record = createRecord(recordid);
    final SolrQueryBuilder builder = new SolrQueryBuilder(getName(), record);

    final AnyMap filter1 = createFilter("v11 v12".split(" "), "testGroup");
    builder.addFilter("att1", filter1);

    final AnyMap filter2 = createFilter("v21 v22".split(" "), "testGroup");
    builder.addFilter("att2", filter2);

    final String expected = "fq=+((att1:v11)(att1:v12))+((att2:v21)(att2:v22))";
    testSolrQueryConverter(recordid, expected);
  }

  public void _test_Filters_And_Facet_GroupName() throws Exception {
    final String recordid = getName();
    final Record record = createRecord(recordid);
    final AnyMap metadata = record.getMetadata();
    final SolrQueryBuilder builder = new SolrQueryBuilder(getName(), record);

    final AnyMap filter1 = createFilter("v11 v12".split(" "), "testGroup");
    builder.addFilter("att1", filter1);

    final AnyMap facetConfig = record.getFactory().createAnyMap();
    facetConfig.put(SolrConstants.FILTER_GROUP, "testGroup");
    facetConfig.put(QueryConstants.ATTRIBUTE, "FacetAtt");
    final AnySeq filters = facetConfig.getSeq(QueryConstants.FILTER_ONEOF, true);
    filters.add("doc");
    filters.add("xsl");
    metadata.getSeq(QueryConstants.FACETBY, true).add(facetConfig);

    final String expected = "fq=+((att1:v11)(att1:v12))+((FacetAtt:doc)(FacetAtt:xsl))";
    testSolrQueryConverter(recordid, expected);
  }

  /**
   * @return
   */
  private AnyMap createFilter(String[] values, String filterGroup) {
    final DataFactory factory = _blackboard.getDataFactory();
    final AnyMap filter = factory.createAnyMap();
    final AnySeq oneOf = factory.createAnySeq();
    for (final Object value : values) {
      oneOf.add(factory.parseFromObject(value));
    }
    filter.put(QueryConstants.FILTER_ONEOF, oneOf);
    filter.put(SolrConstants.FILTER_GROUP, filterGroup);

    return filter;
  }

  public void test_GroupSettings() throws Exception {
    final String recordid = "GroupSettings";
    final Record record = createRecord(recordid);

    final QueryBuilder builder = new QueryBuilder("TestWorkflow", record);
    builder.addGroupByAttribute(SolrConstants.GROUPBY_METHOD_ATTRIBUTES, "TestField", 10);

    final String expected = "group=true&group.field=TestField&group.limit=10";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_GroupSettings_WithGroupName() throws Exception {
    final String recordid = "GroupSettings";
    final Record record = createRecord(recordid);

    final QueryBuilder builder = new QueryBuilder("TestWorkflow", record);
    builder.addGroupByAttribute("TestGroup", SolrConstants.GROUPBY_METHOD_ATTRIBUTES, "TestField", 10);

    final String expected = "group=true&group.field=TestField&group.limit=10";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_GroupSettings_WithGroupNameAndAsMainResult() throws Exception {
    final String recordid = "GroupSettings";
    final Record record = createRecord(recordid);

    final QueryBuilder builder = new QueryBuilder("TestWorkflow", record);
    builder.addGroupByAttribute("TestGroup", SolrConstants.GROUPBY_METHOD_ATTRIBUTES, "TestField", 10, "TestGroup");

    final String expected = "group=true&group.field=TestField&group.limit=10&group.main=true";
    testSolrQueryConverter(recordid, expected);
  }

  public void test_HighlightSetting() throws Exception {
    final String recordid = "HighlightSettings";
    final Record record = createRecord(recordid);
    final QueryBuilder builder = new QueryBuilder(WORKFLOW, record);
    final String[] attributes = { "title", "content" };
    builder.setQuery("test");
    builder.addHighlightByAttribute(attributes);
    final String expected = "q=test&start=0&rows=10&fl=score,_recordid&hl.fl=title&hl.fl=content&hl=true";
    testSolrQueryConverter(recordid, expected);
  }

}
