/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.util;

import static org.apache.commons.lang.StringUtils.join;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * SolrHelper class.
 * 
 * @author pwissel
 * 
 */
public class SolrHelper {

  /**
   * Solr's default instance dir folder.
   */
  private static final String SOLR_INSTANCE_DIR = "conf";

  /**
   * Solr's default data dir folder.
   */
  private static final String SOLR_DATA_DIR = "data";

  /**
   * The solr.xml file.
   */
  private static final String SOLR_XML = "solr.xml";

  /**
   * The solrconfig.xml file.
   */
  private static final String SOLR_CONFIG = "solrconfig.xml";

  /**
   * The schema.xml file.
   */
  private static final String SOLR_SCHEMA = "schema.xml";

  /**
   * The log.
   */
  private final Log _log = LogFactory.getLog(SolrHelper.class);

  /**
   * The SolrProperties.
   */
  private final SolrProperties _properties;

  /**
   * Constructor.
   * 
   * @param properties
   *          the SolrPropertie.
   * @throws IOException
   *           IOException.
   */
  public SolrHelper(final SolrProperties properties) throws IOException {
    _properties = properties;
    initializeWorkspace();
  }

  /**
   * Copy the conf/ of the source core to the workspace for a new core.
   * 
   * @param srcName
   *          the name of the new core.
   * @param targetName
   * @throws IOException
   *           IOException.
   */
  public void copyCoreConfig(String srcName, String targetName) throws IOException {
    File srcDir = getFile(_properties.getWorkspaceFolder().getPath(), srcName, "conf");
    if (!srcDir.isDirectory()) {
      srcDir = getFile(_properties.getConfigurationFolder().getPath(), srcName, "conf");
    }
    if (!srcDir.isDirectory()) {
      throw new FileNotFoundException("source core (folder) doesnt exist: " + srcDir.getAbsolutePath());
    }

    final File destDir = getFile(_properties.getWorkspaceFolder().getPath(), targetName, "conf");
    if (destDir.isDirectory()) {
      throw new IllegalArgumentException("A core directory with the specified name already exists in workspace: "
        + destDir);
    }
    FileUtils.copyDirectory(srcDir, destDir, FileFilterUtils.makeSVNAware(null));
  }

  /**
   * Get File.
   * 
   * @param pathParts
   *          the path parts.
   * @return the file.
   * @throws IOException
   *           IOException.
   */
  private File getFile(String... pathParts) throws IOException {
    final String path = join(pathParts, File.separatorChar);
    return new File(path);
  }

  /**
   * Get the core root dir.
   * 
   * @param core
   *          the core name.
   * @return the core root dir.
   * @throws IOException
   *           IOException.
   */
  public File getCoreRootDir(String core) throws IOException {
    return new File(getSolrHome(), core);
  }

  /**
   * Get the core data dir.
   * 
   * @param core
   *          the core name.
   * @return the core data dir.
   * @throws IOException
   *           IOException.
   */
  public File getCoreDataDir(String core) throws IOException {
    return new File(getCoreRootDir(core), SOLR_DATA_DIR);
  }

  /**
   * Get the core instance dir.
   * 
   * @param core
   *          the core name.
   * @return the core instance dir.
   * @throws IOException
   */
  public File getCoreInstanceDir(String core) throws IOException {
    return new File(getCoreRootDir(core), SOLR_INSTANCE_DIR);
  }

  /**
   * Get the solrconfig.xml by core.
   * 
   * @param core
   *          the core name.
   * @return the path to solrconfig.xml.
   * @throws IOException
   */
  public File getSolrConfigByCore(String core) throws IOException {
    return new File(getCoreInstanceDir(core), SOLR_CONFIG);
  }

  /**
   * Get the schema.xml by core.
   * 
   * @param core
   *          the core name.
   * @return the path to schema.xml.
   * @throws IOException
   *           IOException.
   */
  public File getSolrSchemaByCore(String core) throws IOException {
    return new File(getCoreInstanceDir(core), SOLR_SCHEMA);
  }

  /**
   * Get the solr.home file.
   * 
   * @return the solr.home file.
   * @throws IOException
   *           IOException.
   */
  public File getSolrHome() throws IOException {
    return _properties.getWorkspaceFolder();
  }

  /**
   * Get the solr.xml file.
   * 
   * @return the solr.xml file.
   * @throws IOException
   *           IOException.
   */
  public File getSolrXML() throws IOException {
    return new File(getSolrHome(), SOLR_XML);
  }

  /**
   * Get the path of the data directory relative to solr.home for specified a core.
   * 
   * @param coreName
   *          the core name.
   * @return the path of data directory.
   */
  public String getDataDirPath(String coreName) {
    return coreName + File.separator + SOLR_DATA_DIR;
  }

  /**
   * Initialize the workspace (solr.home) by copying the solr.xml and all folders that seem to be a Core to the
   * workspace.
   * 
   * @throws IOException
   */
  private void initializeWorkspace() throws IOException {
    final String[] solrWorkspaceContent = _properties.getWorkspaceFolder().list();
    if (solrWorkspaceContent != null && solrWorkspaceContent.length > 0) {
      return;
    }

    // copy solr configuration (solr.xml)
    final File configurationFolder = _properties.getConfigurationFolder();
    final File srcFile = new File(configurationFolder, SOLR_XML);
    if (!srcFile.exists()) {
      throw new FileNotFoundException("need for initialization missing file: " + srcFile.getAbsolutePath());
    }
    final File destFile = getSolrXML();
    FileUtils.copyFile(srcFile, destFile);

    /*
     * BETTER: change this simple coping to parsing of solr.xml and copying all those indexes named in there | TM @ May
     * 30, 2011
     */
    final File[] folders = configurationFolder.listFiles((FileFilter) DirectoryFileFilter.DIRECTORY);
    for (File potentialCoreDir : folders) {
      try {
        final File file = new File(potentialCoreDir, "conf");

        final File destDir = getCoreRootDir(potentialCoreDir.getName());
        if (file.isDirectory()) {
          FileUtils.copyDirectory(potentialCoreDir, destDir, FileFilterUtils.makeSVNAware(null));
        }

      } catch (Exception e) {
        _log.warn("failure on copying core config: " + potentialCoreDir);
      }
    } // for

    if (_log.isDebugEnabled()) {
      _log.debug("Solr workspace initialized successfully in solr.home directory: " + getSolrHome());
    }
  }
}
