/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.solr.Activator;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.config.ConfigurationLoadException;
import org.eclipse.smila.utils.workspace.WorkspaceHelper;

/**
 * The SolrProperties class.
 * 
 * @author pwissel
 * 
 */
public class SolrProperties {

  /**
   * The properties file name.
   */
  private static final String PROPERTIES = "solr.properties";

  /**
   * The property embedded.
   */
  public static final String PROP_EMBEDDED = "solr.embedded";

  /**
   * The property workspace folder.
   */
  public static final String PROP_WORKSPACE_FOLDER = "solr.workspaceFolder";

  /**
   * The property server URL.
   */
  public static final String PROP_SERVER_URL = "solr.serverUrl";

  /**
   * The default server URL.
   */
  public static final String DEFAULT_SERVER_URL = "http://localhost:8983/solr";

  /**
   * The log.
   */
  private final Log _log = LogFactory.getLog(SolrProperties.class);

  /**
   * The properties.
   */
  private final Properties _properties;

  /**
   * The configuration folder.
   */
  private File _configurationFolder;

  /**
   * The workspace folder.
   */
  private File _workspaceFolder;

  /**
   * 
   */
  public SolrProperties() {
    this(ConfigUtils.getConfigStream(Activator.BUNDLE_ID, PROPERTIES));
  }

  /**
   * 
   */
  public SolrProperties(Properties solrProps) {
    _properties = solrProps;
  }

  /**
   * Default constructor.
   */
  public SolrProperties(InputStream configStream) {
    try {
      final InputStream stream = configStream;
      _properties = new Properties();
      _properties.load(stream);
    } catch (Exception exception) {
      if (_log.isErrorEnabled()) {
        _log.error("Error while loading solr properties", exception);
      }
      throw new ConfigurationLoadException("Error while loading solr properties", exception);
    }
  }

  /**
   * Is embedded.
   * 
   * @return true if is embedded, false otherwise.
   */
  public Boolean isEmbedded() {
    final Boolean embedded = BooleanUtils.toBooleanObject(_properties.getProperty(PROP_EMBEDDED));
    return BooleanUtils.toBooleanDefaultIfNull(embedded, true);
  }

  /**
   * Get the configuration folder.
   * 
   * @return the configuration folder.
   */
  public File getConfigurationFolder() {
    if (_configurationFolder == null) {
      _configurationFolder = ConfigUtils.getConfigFile(Activator.BUNDLE_ID, "");
    }
    return _configurationFolder;
  }

  /**
   * Get the workspace folder.
   * 
   * @return the workspace folder.
   * @throws IOException
   *           IOException.
   */
  public File getWorkspaceFolder() throws IOException {
    if (_workspaceFolder == null) {
      final String path = _properties.getProperty(PROP_WORKSPACE_FOLDER);
      if (path != null) {
        return _workspaceFolder = new File(path);
      }
      _workspaceFolder = WorkspaceHelper.createWorkingDir(Activator.BUNDLE_ID);
    }
    return _workspaceFolder;
  }

  /**
   * Get the server URL.
   * 
   * @return the server URL.
   */
  public String getServerUrl() {
    if (!isEmbedded()) {
      return StringUtils.defaultString(_properties.getProperty(PROP_SERVER_URL), DEFAULT_SERVER_URL);
    }
    return "No solr server url configured because server is embedded.";
  }

}
