/*******************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. 
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.tika.internal;

import java.io.InputStream;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.tika.language.LanguageIdentifier;
import org.eclipse.smila.common.language.Language;
import org.eclipse.smila.common.language.LanguageIdentifyService;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.osgi.service.component.ComponentContext;

public class TikaLanguageIdentifyService implements LanguageIdentifyService {

  /** name of bundle. */
  public final static String BUNDLE_ID = "org.eclipse.smila.tika";

  /** name of property file. */
  public final static String LANGUAGE_MAPPING = "languageMapping.properties";

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** language mapping, key ISO-639-1 code, value: alternative name. */
  private Properties _langMapping;

  /** OSGI activation method. */
  protected void activate(final ComponentContext context) {
    final InputStream propStream = ConfigUtils.getConfigStream(BUNDLE_ID, LANGUAGE_MAPPING);
    try {
      _langMapping = new Properties();
      _langMapping.load(propStream);
    } catch (final Throwable ex) {
      final String msg = "Error while activating " + BUNDLE_ID;
      _log.error(msg, ex);
      throw new RuntimeException(msg, ex);
    } finally {
      IOUtils.closeQuietly(propStream);
    }
  }

  @Override
  public Language identify(final String content) {
    if (content == null || "".equals(content)) {
      return null;
    }
    final LanguageIdentifier langIdentifier = new LanguageIdentifier(content);
    final String isoLanguage = langIdentifier.getLanguage();
    final boolean isCertain = langIdentifier.isReasonablyCertain();
    String alternativeName = _langMapping.getProperty(isoLanguage);
    if (alternativeName != null && "".equals(alternativeName.trim())) {
      alternativeName = null;
    }
    final Language lang = new Language(isoLanguage, alternativeName, isCertain);
    return lang;
  }

  @Override
  public Set<String> getSupportedLanguages() {
    return LanguageIdentifier.getSupportedLanguages();
  }

  @Override
  public String getAlternativeName(final String isoLanguage) {
    if (isoLanguage == null) {
      return null;
    }
    String alternativeName = _langMapping.getProperty(isoLanguage);
    if (alternativeName != null && "".equals(alternativeName.trim())) {
      alternativeName = null;
    }
    return alternativeName;
  }

}
