/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.workermanager.test;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition.Mode;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.taskworker.test.worker.RecordToRecordWorker;
import org.eclipse.smila.taskworker.test.worker.RunAlwaysWaitWorker;
import org.eclipse.smila.taskworker.test.worker.StreamToStreamWorker;
import org.eclipse.smila.taskworker.test.worker.WaitWorker;
import org.eclipse.smila.test.DeclarativeServiceTestCase;
import org.eclipse.smila.workermanager.ScaleUpControl;
import org.eclipse.smila.workermanager.WorkerManager;
import org.eclipse.smila.workermanager.internal.WorkerManagerImpl;
import org.eclipse.smila.workermanager.keepalive.TaskKeepAlive;

/**
 * Test for WorkerManager class.
 */
public class TestWorkerManager extends DeclarativeServiceTestCase {

  /**
   * tests if WorkerManager service is correctly registered.
   */
  public void testService() throws Exception {
    final WorkerManager workerManager = getService(WorkerManager.class);
    assertNotNull(workerManager);
  }

  /**
   * tests that workers are registered on WorkerManager.
   */
  public void testWorkerRegistration() throws Exception {
    final Worker waitWorker = getService(Worker.class);
    assertNotNull(waitWorker);
    final WorkerManager workerManager = getService(WorkerManager.class);
    assertTrue(workerManager.containsWorker(WaitWorker.NAME));
    assertTrue(workerManager.containsWorker(StreamToStreamWorker.NAME));
    assertTrue(workerManager.containsWorker(RecordToRecordWorker.NAME));
  }

  /**
   * tests that keep alive mechanism in WorkerManager works, resp. that 'canceled' is set in currently processed tasks
   * if keep alive failed.
   */
  public void testWorkerManagerKeepAlive() throws Exception {
    final TaskManager taskManager = getService(TaskManager.class);
    final WorkerManager workerManager = getService(WorkerManager.class);
    final ScaleUpControl scaleUpControl = new ScaleUpControl();
    workerManager.setScaleUpControl(scaleUpControl);
    try {
      workerManager.setKeepAliveInterval(200, 1); // 200 ms for keep alive check, 1 sec for keep alive sent

      // create a task for WaitWorker with processing time of 5 seconds and filter value
      Task task = new Task("1", WaitWorker.NAME);
      task.getParameters().put(WaitWorker.PARAM_TIME_TO_WAIT, "5000");
      task.getProperties().put("myFilter", "value");

      // assure that 5-seconds-task is kept alive
      taskManager.addTasks(Arrays.asList(task));
      Thread.sleep(3000); // after 3 seconds, task can't be finished
      assertEquals(1, scaleUpControl.getTaskCounter(WaitWorker.NAME));
      Thread.sleep(5000); // after another 5 seconds, task should be finished
      assertEquals(0, scaleUpControl.getTaskCounter(WaitWorker.NAME));

      // assure that 5-seconds-task is not kept alive after cancel
      task = new Task("2", WaitWorker.NAME);
      task.getParameters().put(WaitWorker.PARAM_TIME_TO_WAIT, "5000");
      task.getProperties().put("myFilter", "value");
      taskManager.addTasks(Arrays.asList(task));
      Thread.sleep(2000);
      assertEquals(1, scaleUpControl.getTaskCounter(WaitWorker.NAME));
      // simulate cancel - remove task from taskmanager via filter map
      final AnyMap filter = DataFactory.DEFAULT.createAnyMap();
      filter.put("myFilter", "value");
      taskManager.removeTasks(filter);
      Thread.sleep(2000);
      // keep alive should have been called meanwhile -> WaitWorker will abort and finish its task
      assertEquals(0, scaleUpControl.getTaskCounter(WaitWorker.NAME));
    } finally {
      // reset keep alive values
      workerManager.setKeepAliveInterval(TaskKeepAlive.DEFAULT_SCHEDULE_MILLIS,
        TaskKeepAlive.TIME_OUT_DEFAULT_SECONDS);
    }
  }

  /**
   * tests if worker definitions for internal workers are generated automatically.
   */
  public void testInternalWorkerDefinition() throws Exception {
    final WorkerManagerImpl workerManager = (WorkerManagerImpl) getService(WorkerManager.class);
    assertNotNull(workerManager);

    WorkerDefinition workerDef = workerManager.getWorkerDefinition("nonexisting-worker");
    assertNull("Should not get worker defintition for nonexisting worker here", workerDef);

    workerDef = workerManager.getWorkerDefinition(WorkerManager.PREFIX_INTERNAL + "nonexisting-worker");
    assertNotNull("Worker defintition for internal nonexisting worker should be created", workerDef);
    assertTrue(workerDef.getModes().contains(Mode.RUNALWAYS));
  }

  /**
   * tests that runAlways-Worker don't compete with normal workers for scale up seats.
   */
  public void testWorkerManagerRunAlwaysScaleUp() throws Exception {
    final TaskManager taskManager = getService(TaskManager.class);
    final WorkerManager workerManager = getService(WorkerManager.class);
    final ScaleUpControl scaleUpControl = new ScaleUpControl();

    final int maxScaleUp = 8;
    final int noOfTasksToCreate = maxScaleUp * 2;
    final long maxScaleUpForRunAlwaysWorker = 4L;
    scaleUpControl.setScaleUpLimit(RunAlwaysWaitWorker.NAME, maxScaleUpForRunAlwaysWorker);
    // unlimited until max scaleup of node
    scaleUpControl.setScaleUpLimit(WaitWorker.NAME, -1L);
    workerManager.setScaleUpControl(scaleUpControl);
    try {
      workerManager.setKeepAliveInterval(200, 1); // 200 ms for keep alive check, 1 sec for keep alive sent

      // create tasks > keep alive for WaitWorker with processing time of 5 seconds and filter value
      final Collection<Task> tasks = new ArrayList<Task>();
      for (int i = 0; i < noOfTasksToCreate; i++) {
        final Task task = new Task(String.valueOf(i), WaitWorker.NAME);
        task.getParameters().put(WaitWorker.PARAM_TIME_TO_WAIT, "5000");
        task.getProperties().put("myFilter", "value");
        tasks.add(task);
      }

      // create tasks for keepAliveWorker and assure that these, too
      for (int i = 0; i < noOfTasksToCreate; i++) {
        final Task task = new Task(String.valueOf(i), RunAlwaysWaitWorker.NAME);
        task.getParameters().put(WaitWorker.PARAM_TIME_TO_WAIT, "5000");
        task.getProperties().put("myFilter", "value");
        tasks.add(task);
      }
      taskManager.addTasks(tasks);

      // assert that the maximum scaleup has been assigned to workers
      Thread.sleep(1000);
      assertEquals(maxScaleUp, scaleUpControl.getTaskCounter(WaitWorker.NAME));
      assertEquals(maxScaleUpForRunAlwaysWorker, scaleUpControl.getTaskCounter(RunAlwaysWaitWorker.NAME));

      final AnyMap filter = DataFactory.DEFAULT.createAnyMap();
      filter.put("myFilter", "value");
      taskManager.removeTasks(filter);

      Thread.sleep(5000);
      // keep alive should have been called meanwhile -> WaitWorker will abort and finish its task
      assertEquals(0, scaleUpControl.getTaskCounter(WaitWorker.NAME));
    } finally {
      // reset keep alive values
      workerManager.setKeepAliveInterval(TaskKeepAlive.DEFAULT_SCHEDULE_MILLIS,
        TaskKeepAlive.TIME_OUT_DEFAULT_SECONDS);
    }
  }
}
