/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.zookeeper.test;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.eclipse.smila.test.DeclarativeServiceTestCase;
import org.eclipse.smila.zookeeper.ZkConnection;
import org.eclipse.smila.zookeeper.ZooKeeperService;

/**
 * Test for ZkConcurrentMap class.
 * 
 * @author aweber
 */
public class TestZkConnection extends DeclarativeServiceTestCase {

  private ZooKeeperService _zkService;

  private ZkConnection _zk;

  /**
   * get ZooKeeper service.
   * 
   * @throws Exception
   *           no service found.
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
    _zkService = getService(ZooKeeperService.class);
    assertNotNull(_zkService);
    _zk = new ZkConnection(_zkService);
  }

  /**
   * Tests deleting a tree.
   * 
   * @throws Exception
   *           test failed.
   */
  public void testDeleteTree() throws Exception {
    _zk.createPath("/testDeleteTree/1/1/1", ZkConnection.NO_DATA);
    _zk.createPath("/testDeleteTree/1/1/2", ZkConnection.NO_DATA);
    _zk.createPath("/testDeleteTree/1/1/3", ZkConnection.NO_DATA);
    _zk.createPath("/testDeleteTree/1/2/1", ZkConnection.NO_DATA);
    _zk.createPath("/testDeleteTree/1/2/2", ZkConnection.NO_DATA);
    _zk.createPath("/testDeleteTree/2/1/1", ZkConnection.NO_DATA);
    _zk.createPath("/testDeleteTree/2/2/1", ZkConnection.NO_DATA);
    _zk.createPath("/testDeleteTree/2/2/2", ZkConnection.NO_DATA);
    assertNotNull(_zk.exists("/testDeleteTree"));
    assertNotNull(_zk.exists("/testDeleteTree/1/1/1"));
    assertNotNull(_zk.exists("/testDeleteTree/2/2/2"));
    _zk.deleteTree("/testDeleteTree");
    assertNull(_zk.exists("/testDeleteTree/2/2/2"));
    assertNull(_zk.exists("/testDeleteTree/1/1/1"));
    assertNull(_zk.exists("/testDeleteTree"));
  }

  /**
   * test deleting empty tree.
   * 
   * @throws Exception
   *           test failed.
   */
  public void testDeleteEmptyTree() throws Exception {
    _zk.createPath("/testDeleteEmptyTree", ZkConnection.NO_DATA);
    assertNotNull(_zk.exists("/testDeleteEmptyTree"));
    assertTrue(_zk.getChildrenSorted("/testDeleteEmptyTree").isEmpty());
    _zk.deleteTree("/testDeleteEmptyTree");
    assertNull(_zk.exists("/testDeleteEmptyTree"));
  }

  /**
   * test deleting non existing tree.
   * 
   * @throws Exception
   *           test failed.
   */
  public void testDeleteMissingTree() throws Exception {
    assertNull(_zk.exists("/testDeleteMissingTree"));
    _zk.deleteTree("/testDeleteMissingTree");
    assertNull(_zk.exists("/testDeleteMissingTree"));
  }

  /** test ZkConnection.getLeafNodes(). */
  public void testGetLeafNodes() throws Exception {
    _zk.createPath("/testGetLeafNodes" + "/" + "a" + "/" + "leaf1", ZkConnection.NO_DATA);
    _zk.createPath("/testGetLeafNodes" + "/" + "a" + "/" + "leaf2", ZkConnection.NO_DATA);
    _zk.createPath("/testGetLeafNodes" + "/" + "leaf3", ZkConnection.NO_DATA);
    _zk.createPath("/testGetLeafNodes" + "/" + "a" + "/" + "b" + "/" + "leaf4", ZkConnection.NO_DATA);
    final List<String> expectedNodes =
      Arrays.asList("/testGetLeafNodes/a/leaf1", "/testGetLeafNodes/a/leaf2", "/testGetLeafNodes/leaf3",
        "/testGetLeafNodes/a/b/leaf4");
    Collections.sort(expectedNodes);
    final List<String> leafNodes = _zk.getLeafNodes("/testGetLeafNodes");
    Collections.sort(leafNodes);
    assertEquals(expectedNodes, leafNodes);
  }

  public void testAsyncExists() throws Exception {
    final String nodePath = "/" + getName();
    _zk.createNode(nodePath, ZkConnection.NO_DATA);
    assertNotNull(_zk.exists(nodePath, 100));
    assertNull(_zk.exists(nodePath + "/not", 100));
  }
}
