/*******************************************************************************
 * Copyright (c) 2010 Empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Igor Novakovic (Empolis GmbH) - initial implementation
 *******************************************************************************/

package org.eclipse.smila.integration.pipelet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;

/**
 * HelloWorldPipelet is a very simple example of an processing pipelet. Pipelet's configuration (the name of the source
 * and the target attribute) is read from the pipeline directly. This pipelet implements a trivial business logic: Get
 * the value of record's source attribute, append the string " --- Hello world!!!" to it and set the result as a value
 * of the target attribute.
 * */
public class HelloWorldPipelet implements Pipelet {

  /** parameter for source attribute name. */
  private static final String SOURCE_ATT = "source_attribute_name";

  /** parameter for target attribute name. */
  private static final String TARGET_ATT = "target_attribute_name";

  /** the pipelet's configuration. */
  private AnyMap _config;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** {@inheritDoc} */
  @Override
  public String[] process(final Blackboard blackboard, final String[] recordIds) throws ProcessingException {
    final ParameterAccessor paramAccessor = new ParameterAccessor(blackboard, _config);
    final ResultCollector resultCollector =
      new ResultCollector(paramAccessor, _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
    for (final String id : recordIds) {
      try {
        paramAccessor.setCurrentRecord(id);
        String inValue = "";
        String outValue = "";
        // get parameter either from record or configuration via parameter accessor.
        final String sourceAttr = paramAccessor.getRequiredParameter(SOURCE_ATT);
        final String targetAttr = paramAccessor.getRequiredParameter(TARGET_ATT);

        if (blackboard.getMetadata(id).containsKey(sourceAttr)) {
          inValue = blackboard.getMetadata(id).getStringValue(sourceAttr);
        }

        outValue = inValue + " --- Hello world!!!";

        final Value outLiteral = blackboard.getDataFactory().createStringValue(outValue);
        blackboard.getMetadata(id).put(targetAttr, outLiteral);
        resultCollector.addResult(id);
      } catch (final Exception e) {
        resultCollector.addFailedResult(id, e);
      }
    }
    return resultCollector.getResultIds();
  }

  /** {@inheritDoc} */
  @Override
  public void configure(final AnyMap config) throws ProcessingException {
    _config = config;
  }
}
